"use strict";

var _child_process = require("child_process");
var _crypto = require("crypto");
var _dotenv = _interopRequireDefault(require("dotenv"));
var _electron = require("electron");
var _findProcess = _interopRequireDefault(require("find-process"));
var fsPromises = _interopRequireWildcard(require("fs/promises"));
var net = _interopRequireWildcard(require("net"));
var _nodeFs = _interopRequireDefault(require("node:fs"));
var _nodeOs = require("node:os");
var _nodeUtil = require("node:util");
var _os = require("os");
var _path = _interopRequireDefault(require("path"));
var _url = _interopRequireDefault(require("url"));
var _yargs = _interopRequireDefault(require("yargs"));
var _helpers = require("yargs/helpers");
var _i18next = _interopRequireDefault(require("./i18next.config"));
var _pluginManagement = require("./plugin-management");
var _runCmd = require("./runCmd");
var _windowSize = _interopRequireDefault(require("./windowSize"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright 2025 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

let isRunningScript = false;
if (process.env.HEADLAMP_RUN_SCRIPT) {
  isRunningScript = true;
  (0, _runCmd.runScript)();
}
_dotenv.default.config({
  path: _path.default.join(process.resourcesPath, '.env')
});
const isDev = !!process.env.ELECTRON_DEV;
let frontendPath = '';
if (isDev) {
  frontendPath = _path.default.resolve('..', 'frontend', 'build', 'index.html');
} else {
  frontendPath = _path.default.join(process.resourcesPath, 'frontend', 'index.html');
}
const backendToken = (0, _crypto.randomBytes)(32).toString('hex');
const startUrl = (process.env.ELECTRON_START_URL || _url.default.format({
  pathname: frontendPath,
  protocol: 'file:',
  slashes: true
})

// Windows paths use backslashes and for consistency we want to use forward slashes.
// For example: when application triggers refresh it requests a URL with forward slashes and
// we use startUrl to determine if it's an internal or external URL. So it's easier to
// convert everything to forward slashes.
).replace(/\\/g, '/');
const args = (0, _yargs.default)((0, _helpers.hideBin)(process.argv)).command('list-plugins', 'List all static and user-added plugins.', () => {}, () => {
  try {
    const backendPath = _path.default.join(process.resourcesPath, 'headlamp-server');
    const stdout = (0, _child_process.execSync)(`${backendPath} list-plugins`);
    process.stdout.write(stdout);
    process.exit(0);
  } catch (error) {
    console.error(`Error listing plugins: ${error}`);
    process.exit(1);
  }
}).options({
  headless: {
    describe: 'Open Headlamp in the default web browser instead of its app window',
    type: 'boolean'
  },
  'disable-gpu': {
    describe: 'Disable use of GPU. For people who may have buggy graphics drivers',
    type: 'boolean'
  },
  'watch-plugins-changes': {
    describe: 'Reloads plugins when there are changes to them or their directory',
    type: 'boolean'
  },
  port: {
    describe: 'Port for the backend server to listen on',
    type: 'number',
    default: 4466
  }
}).positional('kubeconfig', {
  describe: 'Path to the kube config file (uses the default kube config location if not specified)',
  type: 'string'
}).help().parseSync();
const isHeadlessMode = args.headless === true;
let disableGPU = args['disable-gpu'] === true;
const defaultPort = args.port || 4466;
let actualPort = defaultPort; // Will be updated when backend starts
const MAX_PORT_ATTEMPTS = Math.abs(Number(process.env.HEADLAMP_MAX_PORT_ATTEMPTS) || 100); // Maximum number of ports to try

const useExternalServer = process.env.EXTERNAL_SERVER || false;
const shouldCheckForUpdates = process.env.HEADLAMP_CHECK_FOR_UPDATES !== 'false';

// make it global so that it doesn't get garbage collected
let mainWindow;

/**
 * `Action` is an interface for an action to be performed by the plugin manager.
 *
 * @interface
 * @property {string} identifier - The unique identifier for the action.
 * @property {'INSTALL' | 'UNINSTALL' | 'UPDATE' | 'LIST' | 'CANCEL' | 'GET'} action - The type of the action.
 * @property {string} [URL] - The URL for the action. Optional.
 * @property {string} [destinationFolder] - The destination folder for the action. Optional.
 * @property {string} [headlampVersion] - The version of Headlamp for the action. Optional.
 * @property {string} [pluginName] - The name of the plugin for the action. Optional.
 */

/**
 * `ProgressResp` is an interface for progress response.
 *
 * @interface
 * @property {string} type - The type of the progress response.
 * @property {string} message - The message of the progress response.
 * @property {Record<string, any>} data - Additional data for the progress response. Optional.
 */

/**
 * `PluginManagerEventListeners` is a class that manages event listeners for plugins-manager.
 *
 * @class
 */
class PluginManagerEventListeners {
  cache = {};
  constructor() {
    this.cache = {};
  }

  /**
   * Converts the progress response to a percentage.
   *
   * @param {ProgressResp} progress - The progress response object.
   * @returns {number} The progress as a percentage.
   */
  convertProgressToPercentage(progress) {
    switch (progress.message) {
      case 'Fetching Plugin Metadata':
        return 20;
      case 'Plugin Metadata Fetched':
        return 30;
      case 'Downloading Plugin':
        return 50;
      case 'Plugin Downloaded':
        return 100;
      default:
        return 0;
    }
  }

  /**
   * Sets up event handlers for plugin-manager.
   *
   * @method
   * @name setupEventHandlers
   */
  setupEventHandlers() {
    _electron.ipcMain.on('plugin-manager', async (event, data) => {
      const eventData = JSON.parse(data);
      const {
        identifier,
        action
      } = eventData;
      const updateCache = progress => {
        const percentage = this.convertProgressToPercentage(progress);
        this.cache[identifier].progress = progress;
        this.cache[identifier].percentage = percentage;
      };
      switch (action) {
        case 'INSTALL':
          this.handleInstall(eventData, updateCache);
          break;
        case 'UPDATE':
          this.handleUpdate(eventData, updateCache);
          break;
        case 'UNINSTALL':
          this.handleUninstall(eventData, updateCache);
          break;
        case 'LIST':
          this.handleList(event, eventData);
          break;
        case 'CANCEL':
          this.handleCancel(event, identifier);
          break;
        case 'GET':
          this.handleGet(event, identifier);
          break;
        default:
          console.error(`Unknown action: ${action}`);
      }
    });
  }

  /**
   * Handles the installation process.
   *
   * @method
   * @name handleInstall
   * @private
   */
  async handleInstall(eventData, updateCache) {
    const {
      identifier,
      URL,
      destinationFolder,
      headlampVersion,
      pluginName
    } = eventData;
    if (!mainWindow) {
      return {
        type: 'error',
        message: 'Main window is not available'
      };
    }
    if (!URL) {
      return {
        type: 'error',
        message: 'URL is required'
      };
    }
    const controller = new AbortController();
    this.cache[identifier] = {
      action: 'INSTALL',
      progress: {
        type: 'info',
        message: 'waiting for user consent'
      },
      percentage: 0,
      controller
    };
    let pluginInfo = undefined;
    try {
      pluginInfo = await _pluginManagement.PluginManager.fetchPluginInfo(URL, {
        signal: controller.signal
      });
    } catch (error) {
      console.error('Error fetching plugin info:', error);
      _electron.dialog.showErrorBox(_i18next.default.t('Failed to fetch plugin info'), _i18next.default.t('An error occurred while fetching plugin info from {{  URL }}.', {
        URL
      }));
      return {
        type: 'error',
        message: 'Failed to fetch plugin info'
      };
    }
    const {
      matchingExtraFiles
    } = (0, _pluginManagement.getMatchingExtraFiles)(pluginInfo?.extraFiles ? pluginInfo?.extraFiles : {});
    const extraUrls = matchingExtraFiles.map(file => file.url);
    const allUrls = [pluginInfo.archiveURL, ...extraUrls].join(', ');
    const dialogOptions = {
      type: 'question',
      buttons: [_i18next.default.t('Yes'), _i18next.default.t('No')],
      defaultId: 1,
      title: _i18next.default.t('Plugin Installation'),
      message: _i18next.default.t('Do you want to install the plugin "{{ pluginName }}"?', {
        pluginName
      }),
      detail: _i18next.default.t('You are about to install a plugin from: {{ url }}\nDo you want to proceed?', {
        url: allUrls
      })
    };
    let userChoice;
    try {
      const answer = await _electron.dialog.showMessageBox(mainWindow, dialogOptions);
      userChoice = answer.response;
    } catch (error) {
      console.error('Error during installation process:', error);
      return {
        type: 'error',
        message: 'An error occurred during the installation process'
      };
    }
    console.log('User response:', userChoice);
    if (userChoice === 1) {
      // User clicked "No"
      this.cache[identifier] = {
        action: 'INSTALL',
        progress: {
          type: 'error',
          message: 'installation cancelled due to user consent'
        },
        percentage: 0,
        controller
      };
      return {
        type: 'error',
        message: 'Installation cancelled due to user consent'
      };
    }

    // User clicked "Yes", proceed with installation
    this.cache[identifier] = {
      action: 'INSTALL',
      progress: {
        type: 'info',
        message: 'installing plugin'
      },
      percentage: 10,
      controller
    };
    (0, _runCmd.addRunCmdConsent)(pluginInfo);
    _pluginManagement.PluginManager.installFromPluginPkg(pluginInfo, destinationFolder, headlampVersion, progress => {
      updateCache(progress);
    }, controller.signal);
    return {
      type: 'info',
      message: 'Installation started'
    };
  }
  /**
   * Handles the update process.
   *
   * @method
   * @name handleUpdate
   * @private
   */
  handleUpdate(eventData, updateCache) {
    const {
      identifier,
      pluginName,
      destinationFolder,
      headlampVersion
    } = eventData;
    if (!pluginName) {
      this.cache[identifier] = {
        action: 'UPDATE',
        progress: {
          type: 'error',
          message: 'Plugin Name is required'
        }
      };
      return;
    }
    const controller = new AbortController();
    this.cache[identifier] = {
      action: 'UPDATE',
      percentage: 10,
      progress: {
        type: 'info',
        message: 'updating plugin'
      },
      controller
    };
    _pluginManagement.PluginManager.update(pluginName, destinationFolder, headlampVersion, progress => {
      updateCache(progress);
    }, controller.signal);
  }

  /**
   * Handles the uninstallation process.
   *
   * @method
   * @name handleUninstall
   * @private
   */
  handleUninstall(eventData, updateCache) {
    const {
      identifier,
      pluginName,
      destinationFolder
    } = eventData;
    if (!pluginName) {
      this.cache[identifier] = {
        action: 'UNINSTALL',
        progress: {
          type: 'error',
          message: 'Plugin Name is required'
        }
      };
      return;
    }
    this.cache[identifier] = {
      action: 'UNINSTALL',
      progress: {
        type: 'info',
        message: 'uninstalling plugin'
      }
    };
    (0, _runCmd.removeRunCmdConsent)(pluginName);
    _pluginManagement.PluginManager.uninstall(pluginName, destinationFolder, progress => {
      updateCache(progress);
    });
  }

  /**
   * Handles the list event.
   *
   * Lists plugins from all three directories (shipped, user-installed, development)
   * and returns a combined list with their locations.
   *
   * @method
   * @name handleList
   * @param {Electron.IpcMainEvent} event - The IPC Main Event.
   * @param {Action} eventData - The event data.
   * @private
   */
  handleList(event, eventData) {
    const {
      identifier,
      destinationFolder
    } = eventData;

    // If a specific folder is requested, list only from that folder
    if (destinationFolder) {
      _pluginManagement.PluginManager.list(destinationFolder, progress => {
        event.sender.send('plugin-manager', JSON.stringify({
          identifier: identifier,
          ...progress
        }));
      });
      return;
    }

    // Otherwise, list from all three directories
    try {
      const allPlugins = [];

      // List from shipped plugins (.plugins)
      const shippedDir = _path.default.join(__dirname, '.plugins');
      try {
        const shippedPlugins = _pluginManagement.PluginManager.list(shippedDir);
        if (shippedPlugins) {
          allPlugins.push(...shippedPlugins);
        }
      } catch (error) {
        // Only ignore if directory doesn't exist, log other errors
        if (error?.code !== 'ENOENT') {
          console.error('Error listing shipped plugins:', error);
        }
      }

      // List from user-installed plugins (user-plugins)
      const userDir = (0, _pluginManagement.defaultUserPluginsDir)();
      try {
        const userPlugins = _pluginManagement.PluginManager.list(userDir);
        if (userPlugins) {
          allPlugins.push(...userPlugins);
        }
      } catch (error) {
        // Only ignore if directory doesn't exist, log other errors
        if (error?.code !== 'ENOENT') {
          console.error('Error listing user plugins:', error);
        }
      }

      // List from development plugins (plugins)
      const devDir = (0, _pluginManagement.defaultPluginsDir)();
      try {
        const devPlugins = _pluginManagement.PluginManager.list(devDir);
        if (devPlugins) {
          allPlugins.push(...devPlugins);
        }
      } catch (error) {
        // Only ignore if directory doesn't exist, log other errors
        if (error?.code !== 'ENOENT') {
          console.error('Error listing development plugins:', error);
        }
      }

      // Send combined results
      event.sender.send('plugin-manager', JSON.stringify({
        identifier: identifier,
        type: 'success',
        message: 'Plugins Listed',
        data: allPlugins
      }));
    } catch (error) {
      event.sender.send('plugin-manager', JSON.stringify({
        identifier: identifier,
        type: 'error',
        message: error instanceof Error ? error.message : String(error)
      }));
    }
  }

  /**
   * Handles the cancel event.
   *
   * @method
   * @name handleCancel
   * @param {Electron.IpcMainEvent} event - The IPC Main Event.
   * @param {string} identifier - The identifier of the event to cancel.
   * @private
   */
  handleCancel(event, identifier) {
    const cacheEntry = this.cache[identifier];
    if (cacheEntry?.controller) {
      cacheEntry.controller.abort();
      event.sender.send('plugin-manager', JSON.stringify({
        type: 'success',
        message: 'cancelled'
      }));
    }
  }

  /**
   * Handles the get event.
   *
   * @method
   * @name handleGet
   * @param {Electron.IpcMainEvent} event - The IPC Main Event.
   * @param {string} identifier - The identifier of the event to get.
   * @private
   */
  handleGet(event, identifier) {
    const cacheEntry = this.cache[identifier];
    if (cacheEntry) {
      event.sender.send('plugin-manager', JSON.stringify({
        identifier: identifier,
        ...cacheEntry.progress,
        percentage: cacheEntry.percentage
      }));
    } else {
      event.sender.send('plugin-manager', JSON.stringify({
        type: 'error',
        message: 'No such operation in progress'
      }));
    }
  }
}

/**
 * Returns the user's preferred shell or a fallback shell.
 * @returns A promise that resolves to the shell path.
 */
async function getShell() {
  // Fallback chain
  const shells = ['/bin/zsh', '/bin/bash', '/bin/sh'];
  let userShell = '';
  try {
    userShell = (0, _nodeOs.userInfo)().shell || process.env.SHELL || '';
    if (userShell) shells.unshift(userShell);
  } catch (error) {
    console.error('Failed to get user shell:', error);
  }
  for (const shell of shells) {
    try {
      await fsPromises.stat(shell);
      return shell;
    } catch (error) {
      console.error(`Shell not found: ${shell}, error: ${error}`);
    }
  }
  console.error('No valid shell found, defaulting to /bin/sh');
  return '/bin/sh';
}

/**
 * Retrieves the environment variables from the user's shell.
 * @returns A promise that resolves to the shell environment.
 */
async function getShellEnv() {
  const execPromisify = (0, _nodeUtil.promisify)(_child_process.exec);
  const shell = await getShell();
  const isWindows = process.platform === 'win32';

  // For Windows, just return the current environment
  if (isWindows) {
    return {
      ...process.env
    };
  }

  // For Unix-like systems
  const isZsh = shell.includes('zsh');
  // interactive is supported only on zsh
  const shellArgs = isZsh ? ['--login', '--interactive', '-c'] : ['--login', '-c'];
  try {
    const env = {
      ...process.env,
      DISABLE_AUTO_UPDATE: 'true'
    };
    let stdout;
    let isEnvNull = false;
    try {
      // Try env -0 first
      const command = 'env -0';
      ({
        stdout
      } = await execPromisify(`${shell} ${shellArgs.join(' ')} '${command}'`, {
        encoding: 'utf8',
        timeout: 10000,
        env
      }));
      isEnvNull = true;
    } catch (error) {
      // If env -0 fails, fall back to env
      console.log('env -0 failed, falling back to env');
      const command = 'env';
      ({
        stdout
      } = await execPromisify(`${shell} ${shellArgs.join(' ')} '${command}'`, {
        encoding: 'utf8',
        timeout: 10000,
        env
      }));
    }
    const processLines = separator => {
      return stdout.split(separator).reduce((acc, line) => {
        const firstEqualIndex = line.indexOf('=');
        if (firstEqualIndex > 0) {
          const key = line.slice(0, firstEqualIndex);
          const value = line.slice(firstEqualIndex + 1);
          acc[key] = value;
        }
        return acc;
      }, {});
    };
    const envVars = isEnvNull ? processLines('\0') : processLines('\n');
    const mergedEnv = {
      ...process.env,
      ...envVars
    };
    return mergedEnv;
  } catch (error) {
    console.error('Failed to get shell environment:', error);
    return process.env;
  }
}

/**
 * Check if a port is available by attempting to create a server on it
 */
async function isPortAvailable(port) {
  return new Promise(resolve => {
    const server = net.createServer();
    server.once('error', () => {
      resolve(false);
    });
    server.once('listening', () => {
      server.close(() => resolve(true));
    });
    try {
      server.listen({
        port,
        host: 'localhost',
        exclusive: true
      });
    } catch (err) {
      server.emit('error', err);
    }
  });
}

/**
 * Find an available port starting from the default port
 * Tries to find a free port, skipping all occupied ports (including Headlamp)
 * @returns Available port number, or throws if no port found after MAX_PORT_ATTEMPTS
 */
async function findAvailablePort(startPort) {
  for (let i = 0; i < MAX_PORT_ATTEMPTS; i++) {
    const port = startPort + i;
    // Skip ports already used by another Headlamp instance.
    const headlampPIDs = await getHeadlampPIDsOnPort(port);
    if (headlampPIDs && headlampPIDs.length > 0) {
      console.info(`Port ${port} is occupied by Headlamp process(es) ${headlampPIDs.join(', ')}, trying next port...`);
      continue;
    }
    const available = await isPortAvailable(port);
    if (available) {
      if (port !== startPort) {
        console.info(`Port ${startPort} is in use, using port ${port} instead`);
      }
      return port;
    }
    console.info(`Port ${port} is occupied by another process, trying next port...`);
  }
  throw new Error(`Could not find an available port after ${MAX_PORT_ATTEMPTS} attempts starting from ${startPort}`);
}
async function startServer(flags = []) {
  const serverFilePath = isDev ? _path.default.resolve('../backend/headlamp-server') : _path.default.join(process.resourcesPath, './headlamp-server');
  actualPort = await findAvailablePort(defaultPort);
  let serverArgs = ['--listen-addr=localhost', `--port=${actualPort}`];
  if (!!args.kubeconfig) {
    serverArgs = serverArgs.concat(['--kubeconfig', args.kubeconfig]);
  }
  const manifestDir = isDev ? _path.default.resolve('./') : process.resourcesPath;
  const manifestFile = _path.default.join(manifestDir, 'app-build-manifest.json');
  let buildManifest = {};
  try {
    const manifestContent = await fsPromises.readFile(manifestFile, 'utf8');
    buildManifest = JSON.parse(manifestContent);
  } catch (err) {
    // If the manifest doesn't exist or can't be read, fall back to empty object
    buildManifest = {};
  }
  const proxyUrls = !!buildManifest && buildManifest['proxy-urls'];
  if (!!proxyUrls && proxyUrls.length > 0) {
    serverArgs = serverArgs.concat(['--proxy-urls', proxyUrls.join(',')]);
  }
  if (args.watchPluginsChanges !== undefined) {
    serverArgs.push(`--watch-plugins-changes=${args.watchPluginsChanges}`);
  }
  const bundledPlugins = _path.default.join(process.resourcesPath, '.plugins');

  // Enable the Helm and dynamic cluster endpoints
  process.env.HEADLAMP_CONFIG_ENABLE_HELM = 'true';
  process.env.HEADLAMP_CONFIG_ENABLE_DYNAMIC_CLUSTERS = 'true';

  // Pass a token to the backend that can be used for auth on some routes
  process.env.HEADLAMP_BACKEND_TOKEN = backendToken;

  // Set the bundled plugins in addition to the the user's plugins.
  try {
    const stat = await fsPromises.stat(bundledPlugins);
    if (stat.isDirectory()) {
      const entries = await fsPromises.readdir(bundledPlugins);
      if (entries.length !== 0) {
        process.env.HEADLAMP_STATIC_PLUGINS_DIR = bundledPlugins;
      }
    }
  } catch (err) {
    // Directory doesn't exist or is not readable — ignore and continue.
  }
  serverArgs = serverArgs.concat(flags);
  console.log('arguments passed to backend server', serverArgs);
  let extendedEnv;
  try {
    extendedEnv = await getShellEnv();
  } catch (error) {
    console.error('Failed to get shell environment, using default:', error);
    extendedEnv = process.env;
  }
  const options = {
    detached: true,
    windowsHide: true,
    env: {
      ...extendedEnv
    }
  };
  return (0, _child_process.spawn)(serverFilePath, serverArgs, options);
}

/**
 * Are we running inside WSL?
 * @returns true if we are running inside WSL.
 */
async function isWSL() {
  // Cheap platform check first to avoid reading /proc on non-Linux platforms.
  if ((0, _os.platform)() !== 'linux') {
    return false;
  }
  try {
    const data = await fsPromises.readFile('/proc/version', {
      encoding: 'utf8'
    });
    const lower = data.toLowerCase();
    return lower.includes('microsoft') || lower.includes('wsl');
  } catch {
    return false;
  }
}
let serverProcess;
let intentionalQuit;
let serverProcessQuit;
function quitServerProcess() {
  if ((!serverProcess || serverProcessQuit) && process.platform !== 'win32') {
    console.error('server process already not running');
    return;
  }
  intentionalQuit = true;
  console.info('stopping server process...');
  if (!serverProcess) {
    return;
  }
  serverProcess.stdin.destroy();
  // @todo: should we try and end the process a bit more gracefully?
  //       What happens if the kill signal doesn't kill it?
  serverProcess.kill();
  serverProcess = null;
}
function getAcceleratorForPlatform(navigation) {
  switch ((0, _os.platform)()) {
    case 'darwin':
      return navigation === 'right' ? 'Cmd+]' : 'Cmd+[';
    case 'win32':
      return navigation === 'right' ? 'Alt+Right' : 'Alt+Left';
    default:
      return navigation === 'right' ? 'Alt+Right' : 'Alt+Left';
  }
}
function getDefaultAppMenu() {
  const isMac = process.platform === 'darwin';
  const sep = {
    type: 'separator'
  };
  const aboutMenu = {
    label: _i18next.default.t('About'),
    role: 'about',
    id: 'original-about',
    afterPlugins: true
  };
  const quitMenu = {
    label: _i18next.default.t('Quit'),
    role: 'quit',
    id: 'original-quit'
  };
  const selectAllMenu = {
    label: _i18next.default.t('Select All'),
    role: 'selectAll',
    id: 'original-select-all'
  };
  const deleteMenu = {
    label: _i18next.default.t('Delete'),
    role: 'delete',
    id: 'original-delete'
  };
  const appMenu = [
  // { role: 'appMenu' }
  ...(isMac ? [{
    label: _electron.app.name,
    submenu: [aboutMenu, sep, {
      label: _i18next.default.t('Services'),
      role: 'services',
      id: 'original-services'
    }, sep, {
      label: _i18next.default.t('Hide'),
      role: 'hide',
      id: 'original-hide'
    }, {
      label: _i18next.default.t('Hide Others'),
      role: 'hideothers',
      id: 'original-hide-others'
    }, {
      label: _i18next.default.t('Show All'),
      role: 'unhide',
      id: 'original-show-all'
    }, sep, quitMenu]
  }] : []),
  // { role: 'fileMenu' }
  {
    label: _i18next.default.t('File'),
    id: 'original-file',
    submenu: [isMac ? {
      label: _i18next.default.t('Close'),
      role: 'close',
      id: 'original-close'
    } : quitMenu]
  },
  // { role: 'editMenu' }
  {
    label: _i18next.default.t('Edit'),
    id: 'original-edit',
    submenu: [{
      label: _i18next.default.t('Cut'),
      role: 'cut',
      id: 'original-cut'
    }, {
      label: _i18next.default.t('Copy'),
      role: 'copy',
      id: 'original-copy'
    }, {
      label: _i18next.default.t('Paste'),
      role: 'paste',
      id: 'original-paste'
    }, ...(isMac ? [{
      label: _i18next.default.t('Paste and Match Style'),
      role: 'pasteAndMatchStyle',
      id: 'original-paste-and-match-style'
    }, deleteMenu, selectAllMenu, sep, {
      label: _i18next.default.t('Speech'),
      id: 'original-speech',
      submenu: [{
        label: _i18next.default.t('Start Speaking'),
        role: 'startspeaking',
        id: 'original-start-speaking'
      }, {
        label: _i18next.default.t('Stop Speaking'),
        role: 'stopspeaking',
        id: 'original-stop-speaking'
      }]
    }] : [deleteMenu, sep, selectAllMenu])]
  },
  // { role: 'viewMenu' }
  {
    label: _i18next.default.t('View'),
    id: 'original-view',
    submenu: [{
      label: _i18next.default.t('Toggle Developer Tools'),
      role: 'toggledevtools',
      id: 'original-toggle-dev-tools'
    }, sep, {
      label: _i18next.default.t('Reset Zoom'),
      id: 'original-reset-zoom',
      accelerator: 'CmdOrCtrl+0',
      click: () => setZoom(1.0)
    }, {
      label: _i18next.default.t('Zoom In'),
      id: 'original-zoom-in',
      accelerator: 'CmdOrCtrl+Plus',
      click: () => adjustZoom(0.1)
    }, {
      label: _i18next.default.t('Zoom Out'),
      id: 'original-zoom-out',
      accelerator: 'CmdOrCtrl+-',
      click: () => adjustZoom(-0.1)
    }, sep, {
      label: _i18next.default.t('Toggle Fullscreen'),
      role: 'togglefullscreen',
      id: 'original-toggle-fullscreen'
    }]
  }, {
    label: _i18next.default.t('Navigate'),
    id: 'original-navigate',
    submenu: [{
      label: _i18next.default.t('Reload'),
      role: 'forcereload',
      id: 'original-force-reload'
    }, sep, {
      label: _i18next.default.t('Go to Home'),
      role: 'homescreen',
      id: 'original-home-screen',
      click: () => {
        mainWindow?.loadURL(startUrl);
      }
    }, {
      label: _i18next.default.t('Go Back'),
      role: 'back',
      id: 'original-back',
      accelerator: getAcceleratorForPlatform('left'),
      enabled: false,
      click: () => {
        mainWindow?.webContents.goBack();
      }
    }, {
      label: _i18next.default.t('Go Forward'),
      role: 'forward',
      id: 'original-forward',
      accelerator: getAcceleratorForPlatform('right'),
      enabled: false,
      click: () => {
        mainWindow?.webContents.goForward();
      }
    }]
  }, {
    label: _i18next.default.t('Window'),
    id: 'original-window',
    submenu: [{
      label: _i18next.default.t('Minimize'),
      role: 'minimize',
      id: 'original-minimize'
    }, ...(isMac ? [sep, {
      label: _i18next.default.t('Bring All to Front'),
      role: 'front',
      id: 'original-front'
    }, sep, {
      label: _i18next.default.t('Window'),
      role: 'window',
      id: 'original-window'
    }] : [{
      label: _i18next.default.t('Close'),
      role: 'close',
      id: 'original-close'
    }])]
  }, {
    label: _i18next.default.t('Help'),
    role: 'help',
    id: 'original-help',
    afterPlugins: true,
    submenu: [{
      label: _i18next.default.t('Documentation'),
      id: 'original-documentation',
      url: 'https://headlamp.dev/docs/latest/'
    }, {
      label: _i18next.default.t('Open an Issue'),
      id: 'original-open-issue',
      url: 'https://github.com/kubernetes-sigs/headlamp/issues'
    }, {
      label: _i18next.default.t('About'),
      id: 'original-about-help'
    }]
  }];
  return appMenu;
}
let loadFullMenu = false;
let currentMenu = [];
function setMenu(appWindow, newAppMenu = []) {
  let appMenu = newAppMenu;
  if (appMenu?.length === 0) {
    appMenu = getDefaultAppMenu();
  }
  let menu;
  try {
    const menuTemplate = menusToTemplate(appWindow, appMenu) || [];
    menu = _electron.Menu.buildFromTemplate(menuTemplate);
  } catch (e) {
    console.error(`Failed to build menus from template ${appMenu}:`, e);
    return;
  }
  currentMenu = appMenu;
  _electron.Menu.setApplicationMenu(menu);
}
function updateMenuLabels(menus) {
  let menusToProcess = getDefaultAppMenu();
  const defaultMenusObj = {};

  // Add all default menus in top levels and in submenus to an object:
  // id -> menu.
  while (menusToProcess.length > 0) {
    const menu = menusToProcess.shift();
    // Do not process menus that have no ids, otherwise we cannot be
    // sure which one is which.
    if (!menu.id) {
      continue;
    }
    defaultMenusObj[menu.id] = menu;
    if (menu.submenu) {
      menusToProcess = [...menusToProcess, ...menu.submenu];
    }
  }

  // Add all current menus in top levels and in submenus to a list.
  menusToProcess = [...menus];
  const menusList = [];
  while (menusToProcess.length > 0) {
    const menu = menusToProcess.shift();
    menusList.push(menu);
    if (menu.submenu) {
      menusToProcess = [...menusToProcess, ...menu.submenu];
    }
  }

  // Replace all labels with default labels if the default and current
  // menu ids are the same.
  menusList.forEach(menu => {
    if (!!menu.label && defaultMenusObj[menu.id]) {
      menu.label = defaultMenusObj[menu.id].label;
    }
  });
}
function menusToTemplate(mainWindow, menusFromPlugins) {
  const menusToDisplay = [];
  menusFromPlugins.forEach(appMenu => {
    const {
      url,
      afterPlugins = false,
      ...otherProps
    } = appMenu;
    const menu = otherProps;
    if (!loadFullMenu && !!afterPlugins) {
      return;
    }

    // Handle the "About" menu item from the Help menu specially
    if (appMenu.id === 'original-about-help') {
      menu.click = () => {
        mainWindow?.webContents.send('open-about-dialog');
      };
    } else if (!!url) {
      menu.click = async () => {
        // Open external links in the external browser.
        if (!!mainWindow && !url.startsWith('http')) {
          mainWindow.webContents.loadURL(url);
        } else {
          await _electron.shell.openExternal(url);
        }
      };
    }

    // If the menu has a submenu, then recursively convert it.
    if (Array.isArray(otherProps.submenu)) {
      menu.submenu = menusToTemplate(mainWindow, otherProps.submenu);
    }
    menusToDisplay.push(menu);
  });
  return menusToDisplay;
}
async function getRunningHeadlampPIDs() {
  const processes = await (0, _findProcess.default)('name', 'headlamp-server.*');
  if (processes.length === 0) {
    return null;
  }
  return processes.map(pInfo => pInfo.pid);
}

/**
 * Check if a specific port is occupied by a Headlamp process
 * @returns Array of Headlamp PIDs using the port, or null if port is free or used by another process
 */
async function getHeadlampPIDsOnPort(port) {
  try {
    // Get all Headlamp processes
    const headlampProcesses = await (0, _findProcess.default)('name', 'headlamp-server');
    if (headlampProcesses.length === 0) {
      return null;
    }

    // Parse command line arguments to find which Headlamp process is using this port
    const headlampOnPort = headlampProcesses.filter(p => {
      if (!p.cmd) return false;

      // Look for --port=XXXX or --port XXXX in the command line
      const portRegex = /--port[=\s]+(\d+)/;
      const match = p.cmd.match(portRegex);
      if (match && match[1]) {
        const processPort = parseInt(match[1], 10);
        return processPort === port;
      }

      // If no port specified, Headlamp uses default port 4466
      if (port === 4466 && !p.cmd.includes('--port')) {
        return true;
      }
      return false;
    });
    if (headlampOnPort.length === 0) {
      return null;
    }
    return headlampOnPort.map(p => p.pid);
  } catch (error) {
    console.error(`Error checking if port ${port} is used by Headlamp:`, error);
    return null;
  }
}
function killProcess(pid) {
  if (process.platform === 'win32') {
    // Otherwise on Windows the process will stick around.
    (0, _child_process.execSync)('taskkill /pid ' + pid + ' /T /F');
  } else {
    process.kill(pid, 'SIGHUP');
  }
}
const ZOOM_FILE_PATH = _path.default.join(_electron.app.getPath('userData'), 'headlamp-config.json');
let cachedZoom = 1.0;
function saveZoomFactor(factor) {
  try {
    _nodeFs.default.writeFileSync(ZOOM_FILE_PATH, JSON.stringify({
      zoomFactor: factor
    }), 'utf-8');
  } catch (err) {
    console.error('Failed to save zoom factor:', err);
  }
}
async function loadZoomFactor() {
  try {
    const content = await fsPromises.readFile(ZOOM_FILE_PATH, 'utf-8');
    const {
      zoomFactor = 1.0
    } = JSON.parse(content);
    return typeof zoomFactor === 'number' ? zoomFactor : 1.0;
  } catch (err) {
    console.error('Failed to load zoom factor, defaulting to 1.0:', err);
    return 1.0;
  }
}

// The zoom factor should respect the fixed limits set by Electron.
function clampZoom(factor) {
  return Math.min(5.0, Math.max(0.25, factor));
}
function setZoom(factor) {
  cachedZoom = factor;
  mainWindow?.webContents.setZoomFactor(cachedZoom);
}
function adjustZoom(delta) {
  const newZoom = clampZoom(cachedZoom + delta);
  setZoom(newZoom);
}
function startElectron() {
  console.info('App starting...');

  // Increase max listeners to prevent false positive warnings
  // The app legitimately needs multiple IPC listeners (currently 11)
  // Default is 10, setting to 20 provides headroom for future additions
  _electron.ipcMain.setMaxListeners(20);
  let appVersion;
  if (isDev && process.env.HEADLAMP_APP_VERSION) {
    appVersion = process.env.HEADLAMP_APP_VERSION;
    console.debug(`Overridding app version to ${appVersion}`);
  } else {
    appVersion = _electron.app.getVersion();
  }
  console.log('Check for updates: ', shouldCheckForUpdates);
  async function startServerIfNeeded() {
    if (!useExternalServer) {
      try {
        // Try to start the server (it will find an available port)
        serverProcess = await startServer();
        attachServerEventHandlers(serverProcess);
        serverProcess.addListener('exit', async e => {
          const ERROR_ADDRESS_IN_USE = 98;
          if (e === ERROR_ADDRESS_IN_USE) {
            // This is a fallback - we should have already checked for port conflicts
            // before starting the server. This handles edge cases where the port
            // became occupied between our check and the server start.
            console.warn('Server failed to start due to address in use (unexpected)');
            const runningHeadlamp = await getRunningHeadlampPIDs();
            if (!mainWindow) {
              return;
            }
            if (!!runningHeadlamp) {
              const resp = _electron.dialog.showMessageBoxSync(mainWindow, {
                title: _i18next.default.t('Another process is running'),
                message: _i18next.default.t('Looks like another process is already running. Continue by terminating that process automatically, or quit?'),
                type: 'question',
                buttons: [_i18next.default.t('Continue'), _i18next.default.t('Quit')]
              });
              if (resp === 0) {
                runningHeadlamp.forEach(pid => {
                  try {
                    killProcess(pid);
                  } catch (e) {
                    const message = e instanceof Error ? e.message : String(e);
                    console.error(`Failed to kill process with PID ${pid}: ${message}`);
                  }
                });

                // Wait a bit and retry
                await new Promise(resolve => setTimeout(resolve, 1000));
              } else {
                mainWindow.close();
                return;
              }
            }

            // If we couldn't kill the process, warn the user and quit.
            const processes = await getRunningHeadlampPIDs();
            if (!!processes) {
              _electron.dialog.showMessageBoxSync({
                type: 'warning',
                title: _i18next.default.t('Failed to quit the other running process'),
                message: _i18next.default.t(`Could not quit the other running process, PIDs: {{ process_list }}. Please stop that process and relaunch the app.`, {
                  process_list: processes
                })
              });
              mainWindow.close();
              return;
            }
            serverProcess = await startServer();
            attachServerEventHandlers(serverProcess);
          }
        });
      } catch (error) {
        // Failed to find an available port after all attempts
        const message = error instanceof Error ? error.message : String(error);
        console.error('Failed to find an available port:', message);
        if (!mainWindow) {
          console.error('Cannot show dialog - no main window available');
          return;
        }

        // Ask user if they want to kill existing Headlamp processes and retry
        const headlampPIDs = await getRunningHeadlampPIDs();
        if (headlampPIDs && headlampPIDs.length > 0) {
          const resp = _electron.dialog.showMessageBoxSync(mainWindow, {
            title: _i18next.default.t('No available ports'),
            message: _i18next.default.t('Could not find an available port. There are processes running on ports {{startPort}}-{{endPort}}. Terminate these processes and retry?', {
              startPort: defaultPort,
              endPort: defaultPort + MAX_PORT_ATTEMPTS - 1
            }),
            type: 'warning',
            buttons: [_i18next.default.t('Terminate and Retry'), _i18next.default.t('Quit')]
          });
          if (resp === 0) {
            // User chose to terminate and retry
            console.info(`Terminating ${headlampPIDs.length} Headlamp process(es)...`);
            headlampPIDs.forEach(pid => {
              try {
                killProcess(pid);
              } catch (e) {
                const killMessage = e instanceof Error ? e.message : String(e);
                console.error(`Failed to kill headlamp-server process with PID ${pid}: ${killMessage}`);
              }
            });

            // Wait for processes to be killed
            await new Promise(resolve => setTimeout(resolve, 1000));

            // Retry starting the server
            try {
              serverProcess = await startServer();
              attachServerEventHandlers(serverProcess);
            } catch (retryError) {
              const retryMessage = retryError instanceof Error ? retryError.message : String(retryError);
              console.error('Failed to start server after killing processes:', retryMessage);
              _electron.dialog.showErrorBox(_i18next.default.t('Failed to start'), _i18next.default.t('Could not start the server even after terminating existing processes.'));
              mainWindow.close();
            }
          } else {
            // User chose to quit
            mainWindow.close();
          }
        } else {
          // No Headlamp processes found, but still can't find a port
          _electron.dialog.showErrorBox(_i18next.default.t('No available ports'), _i18next.default.t('Could not find an available port in the range {{startPort}}-{{endPort}}. Please free up a port and try again.', {
            startPort: defaultPort,
            endPort: defaultPort + MAX_PORT_ATTEMPTS - 1
          }));
          mainWindow.close();
        }
      }
    }
  }
  async function createWindow() {
    // WSL has a problem with full size window placement, so make it smaller.
    const withMargin = await isWSL();
    const {
      width,
      height
    } = (0, _windowSize.default)(_electron.screen.getPrimaryDisplay().workAreaSize, withMargin);
    mainWindow = new _electron.BrowserWindow({
      width,
      height,
      webPreferences: {
        nodeIntegration: false,
        contextIsolation: true,
        preload: `${__dirname}/preload.js`
      }
    });

    // Load the frontend
    mainWindow.loadURL(startUrl);
    setMenu(mainWindow, currentMenu);
    mainWindow.webContents.setWindowOpenHandler(({
      url
    }) => {
      // allow all urls starting with app startUrl to open in electron
      if (url.startsWith(startUrl)) {
        return {
          action: 'allow'
        };
      }
      // otherwise open url in a browser and prevent default
      _electron.shell.openExternal(url);
      return {
        action: 'deny'
      };
    });
    mainWindow.webContents.on('did-start-navigation', () => {
      const navigateMenu = _electron.Menu.getApplicationMenu()?.getMenuItemById('original-navigate')?.submenu;
      const goBackMenu = navigateMenu?.getMenuItemById('original-back');
      if (!!goBackMenu) {
        goBackMenu.enabled = mainWindow?.webContents.canGoBack() || false;
      }
      const goForwardMenu = navigateMenu?.getMenuItemById('original-forward');
      if (!!goForwardMenu) {
        goForwardMenu.enabled = mainWindow?.webContents.canGoForward() || false;
      }
    });
    mainWindow.webContents.on('did-finish-load', async () => {
      const startZoom = await loadZoomFactor();
      if (startZoom !== 1.0) {
        setZoom(startZoom);
      }

      // Inject the backend port into the window object
      mainWindow?.webContents.executeJavaScript(`window.headlampBackendPort = ${actualPort};`);
    });
    mainWindow.webContents.on('dom-ready', () => {
      const defaultMenu = getDefaultAppMenu();
      const currentMenu = JSON.parse(JSON.stringify(defaultMenu));
      mainWindow?.webContents.send('currentMenu', currentMenu);
    });
    mainWindow.on('closed', () => {
      mainWindow = null;
    });

    // Workaround to cookies to be saved, since file:// protocal and localhost:port
    // are treated as a cross site request.
    mainWindow.webContents.session.webRequest.onHeadersReceived((details, callback) => {
      if (details.url.startsWith(`http://localhost:${actualPort}`)) {
        callback({
          responseHeaders: {
            ...details.responseHeaders,
            'Set-Cookie': details.responseHeaders?.['Set-Cookie']?.map(it => it.replace('SameSite=Strict', 'SameSite=None;Secure=true')) ?? []
          }
        });
      } else {
        callback(details);
      }
    });

    // Force Single Instance Application
    const gotTheLock = _electron.app.requestSingleInstanceLock();
    if (gotTheLock) {
      _electron.app.on('second-instance', () => {
        // Someone tried to run a second instance, we should focus our window.
        if (mainWindow) {
          if (mainWindow.isMinimized()) mainWindow.restore();
          mainWindow.focus();
        }
      });
    } else {
      _electron.app.quit();
      return;
    }

    /*
    if a library is trying to open a url other than app url in electron take it
    to the default browser
    */
    mainWindow.webContents.on('will-navigate', (event, encodedUrl) => {
      const url = decodeURI(encodedUrl);
      if (url.startsWith(startUrl)) {
        return;
      }
      event.preventDefault();
      _electron.shell.openExternal(url);
    });
    _electron.app.on('open-url', (event, url) => {
      mainWindow?.focus();
      let urlObj;
      try {
        urlObj = new URL(url);
      } catch (e) {
        _electron.dialog.showErrorBox(_i18next.default.t('Invalid URL'), _i18next.default.t('Application opened with an invalid URL: {{ url }}', {
          url
        }));
        return;
      }
      const urlParam = urlObj.hostname;
      let baseUrl = startUrl;
      // this check helps us to avoid adding multiple / to the startUrl when appending the incoming url to it
      if (baseUrl.endsWith('/')) {
        baseUrl = baseUrl.slice(0, startUrl.length - 1);
      }
      // load the index.html from build and route to the hostname received in the protocol handler url
      mainWindow?.loadURL(baseUrl + '#' + urlParam + urlObj.search);
    });
    _i18next.default.on('languageChanged', () => {
      updateMenuLabels(currentMenu);
      setMenu(mainWindow, currentMenu);
    });
    _electron.ipcMain.on('appConfig', () => {
      mainWindow?.webContents.send('appConfig', {
        checkForUpdates: shouldCheckForUpdates,
        appVersion
      });
    });
    _electron.ipcMain.on('pluginsLoaded', () => {
      loadFullMenu = true;
      console.info('Plugins are loaded. Loading full menu.');
      setMenu(mainWindow, currentMenu);
    });
    _electron.ipcMain.on('setMenu', (event, menus) => {
      if (!mainWindow) {
        return;
      }

      // We don't even process this call if we're running in headless mode.
      if (isHeadlessMode) {
        console.log('Ignoring menu change from plugins because of headless mode.');
        return;
      }

      // Ignore the menu change if we received null.
      if (!menus) {
        console.log('Ignoring menu change from plugins because null was sent.');
        return;
      }

      // We update the menu labels here in case the language changed between the time
      // the original menu was sent to the renderer and the time it was received here.
      updateMenuLabels(menus);
      setMenu(mainWindow, menus);
    });
    _electron.ipcMain.on('locale', (event, newLocale) => {
      if (!!newLocale && _i18next.default.language !== newLocale) {
        _i18next.default.changeLanguage(newLocale);
      }
    });
    _electron.ipcMain.on('request-backend-token', () => {
      mainWindow?.webContents.send('backend-token', backendToken);
    });
    _electron.ipcMain.on('request-backend-port', () => {
      mainWindow?.webContents.send('backend-port', actualPort);
    });
    (0, _runCmd.setupRunCmdHandlers)(mainWindow, _electron.ipcMain);
    new PluginManagerEventListeners().setupEventHandlers();

    // Handle opening plugin folder in file explorer
    _electron.ipcMain.on('open-plugin-folder', (event, pluginInfo) => {
      let folderPath = null;
      if (pluginInfo.type === 'user') {
        folderPath = _path.default.join((0, _pluginManagement.defaultUserPluginsDir)(), pluginInfo.folderName);
      } else if (pluginInfo.type === 'development') {
        folderPath = _path.default.join((0, _pluginManagement.defaultPluginsDir)(), pluginInfo.folderName);
      } else if (pluginInfo.type === 'shipped') {
        folderPath = _path.default.join(process.resourcesPath, '.plugins', pluginInfo.folderName);
      }
      if (folderPath) {
        _electron.shell.openPath(folderPath).catch(err => {
          console.error('Failed to open plugin folder:', err);
        });
      }
    });

    // Also add bundled plugin bin directories to PATH
    const bundledPlugins = _path.default.join(process.resourcesPath, '.plugins');
    const bundledPluginBinDirs = (0, _pluginManagement.getPluginBinDirectories)(bundledPlugins);
    if (bundledPluginBinDirs.length > 0) {
      (0, _pluginManagement.addToPath)(bundledPluginBinDirs, 'bundled plugin');
    }

    // Add the installed plugins as well
    const userPluginBinDirs = (0, _pluginManagement.getPluginBinDirectories)((0, _pluginManagement.defaultUserPluginsDir)());
    if (userPluginBinDirs.length > 0) {
      (0, _pluginManagement.addToPath)(userPluginBinDirs, 'userPluginBinDirs plugin');
    }
  }
  if (disableGPU) {
    console.info('Disabling GPU hardware acceleration. Reason: related flag is set.');
  } else if (disableGPU === undefined && process.platform === 'linux' && ['arm', 'arm64'].includes(process.arch)) {
    console.info('Disabling GPU hardware acceleration. Reason: known graphical issues in Linux on ARM (use --disable-gpu=false to force it if needed).');
    disableGPU = true;
  }
  if (disableGPU) {
    _electron.app.disableHardwareAcceleration();
  }
  _electron.app.on('ready', async () => {
    await Promise.all([startServerIfNeeded(), createWindow()]);
  });
  _electron.app.on('activate', async function () {
    if (mainWindow === null) {
      await Promise.all([startServerIfNeeded(), createWindow()]);
    }
  });
  _electron.app.once('window-all-closed', _electron.app.quit);
  _electron.app.once('before-quit', () => {
    saveZoomFactor(cachedZoom);
    _i18next.default.off('languageChanged');
    if (mainWindow) {
      mainWindow.removeAllListeners('close');
    }
  });
}
if (!isRunningScript) {
  _electron.app.on('quit', quitServerProcess);
}

/**
 * add some error handlers to the serverProcess.
 * @param  {ChildProcess} serverProcess to attach the error handlers to.
 */
function attachServerEventHandlers(serverProcess) {
  serverProcess.on('error', err => {
    console.error(`server process failed to start: ${err}`);
  });
  const extractPortFromOutput = data => {
    const output = data.toString();
    const portMatch = output.match(/Listen address:.*:(\d+)/);
    if (portMatch && portMatch[1]) {
      actualPort = parseInt(portMatch[1], 10);
      console.info(`Backend server listening on port: ${actualPort}`);

      // Update the environment variable for the frontend
      if (mainWindow) {
        mainWindow.webContents.executeJavaScript(`window.headlampBackendPort = ${actualPort};`);
      }
    }
  };
  serverProcess.stdout.on('data', data => {
    console.info(`server process stdout: ${data}`);
    extractPortFromOutput(data);
  });
  serverProcess.stderr.on('data', data => {
    const sterrMessage = `server process stderr: ${data}`;
    if (data && data.indexOf && data.indexOf('Requesting') !== -1) {
      // The server prints out urls it's getting, which aren't errors.
      console.info(sterrMessage);
    } else {
      console.error(sterrMessage);
    }
    extractPortFromOutput(data);
  });
  serverProcess.on('close', (code, signal) => {
    const closeMessage = `server process process exited with code:${code} signal:${signal}`;
    if (!intentionalQuit) {
      // @todo: message mainWindow, or loadURL to an error url?
      console.error(closeMessage);
    } else {
      console.info(closeMessage);
    }
    serverProcessQuit = true;
  });
}
if (isHeadlessMode) {
  startServer(['-html-static-dir', _path.default.join(process.resourcesPath, './frontend')]).then(serverProcess => {
    attachServerEventHandlers(serverProcess);

    // Give 1s for backend to start
    setTimeout(() => _electron.shell.openExternal(`http://localhost:${actualPort}`), 1000);
  });
} else {
  if (!isRunningScript) {
    startElectron();
  }
}