"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = envPaths;
var _nodeOs = _interopRequireDefault(require("node:os"));
var _nodePath = _interopRequireDefault(require("node:path"));
var _nodeProcess = _interopRequireDefault(require("node:process"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright 2025 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
From https://github.com/sindresorhus/env-paths

MIT License

Copyright (c) Sindre Sorhus <sindresorhus@gmail.com> (https://sindresorhus.com)

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

const homedir = _nodeOs.default.homedir();
const tmpdir = _nodeOs.default.tmpdir();
const {
  env
} = _nodeProcess.default;
function macos(name) {
  const library = _nodePath.default.join(homedir, 'Library');
  return {
    data: _nodePath.default.join(library, 'Application Support', name),
    config: _nodePath.default.join(library, 'Preferences', name),
    cache: _nodePath.default.join(library, 'Caches', name),
    log: _nodePath.default.join(library, 'Logs', name),
    temp: _nodePath.default.join(tmpdir, name)
  };
}
function windows(name) {
  const appData = env.APPDATA || _nodePath.default.join(homedir, 'AppData', 'Roaming');
  const localAppData = env.LOCALAPPDATA || _nodePath.default.join(homedir, 'AppData', 'Local');
  return {
    // Data/config/cache/log are invented by me as Windows isn't opinionated about this
    data: _nodePath.default.join(localAppData, name, 'Data'),
    config: _nodePath.default.join(appData, name, 'Config'),
    cache: _nodePath.default.join(localAppData, name, 'Cache'),
    log: _nodePath.default.join(localAppData, name, 'Log'),
    temp: _nodePath.default.join(tmpdir, name)
  };
}

// https://specifications.freedesktop.org/basedir-spec/basedir-spec-latest.html
function linux(name) {
  const username = _nodePath.default.basename(homedir);
  return {
    data: _nodePath.default.join(env.XDG_DATA_HOME || _nodePath.default.join(homedir, '.local', 'share'), name),
    config: _nodePath.default.join(env.XDG_CONFIG_HOME || _nodePath.default.join(homedir, '.config'), name),
    cache: _nodePath.default.join(env.XDG_CACHE_HOME || _nodePath.default.join(homedir, '.cache'), name),
    // https://wiki.debian.org/XDGBaseDirectorySpecification#state
    log: _nodePath.default.join(env.XDG_STATE_HOME || _nodePath.default.join(homedir, '.local', 'state'), name),
    temp: _nodePath.default.join(tmpdir, username, name)
  };
}
function envPaths(name, {
  suffix = 'nodejs'
} = {}) {
  if (typeof name !== 'string') {
    throw new TypeError(`Expected a string, got ${typeof name}`);
  }

  // Add suffix to prevent possible conflict with native apps
  const theName = suffix ? `${name}-${suffix}` : name;
  if (_nodeProcess.default.platform === 'darwin') {
    return macos(theName);
  }
  if (_nodeProcess.default.platform === 'win32') {
    return windows(theName);
  }
  return linux(theName);
}