import { HttpOperationResponse, RequestOptionsBase, RestError, WebResource } from "@azure/ms-rest-js";
import { AzureServiceClient } from "./azureServiceClient";
import { LongRunningOperationStates } from "./util/constants";
export declare type LROPollStrategyType = "AzureAsyncOperation" | "Location" | "GetResource";
export interface LROPollState {
    pollStrategyType: LROPollStrategyType;
    initialResponse: HttpOperationResponse;
    state: LongRunningOperationStates;
    mostRecentRequest: WebResource;
    mostRecentResponse: HttpOperationResponse;
    resource: any;
    azureAsyncOperationHeaderValue?: string;
    locationHeaderValue?: string;
    options?: RequestOptionsBase;
}
/**
 * A long-running operation polling strategy base class that other polling strategies should extend.
 */
export declare abstract class LROPollStrategy {
    private readonly _azureServiceClient;
    protected readonly _pollState: LROPollState;
    constructor(_azureServiceClient: AzureServiceClient, _pollState: LROPollState);
    getOperationStatus(): LongRunningOperationStates;
    /**
     * Get whether or not this poll strategy's LRO is finished.
     * @returns Whether or not this poll strategy's LRO is finished.
     */
    isFinished(): boolean;
    /**
     * Send poll requests that check the LRO's status until it is determined that the LRO is finished.
     * @returns Whether or not the LRO succeeded.
     */
    pollUntilFinished(): Promise<boolean>;
    /**
     * Send a single poll request that checks the LRO's status and return the response. If the LRO is
     * finished, then no request will be sent and the last response received will be returned.
     */
    abstract sendPollRequest(): Promise<void>;
    abstract isFinalStatusAcceptable(): boolean;
    protected shouldDoFinalGetResourceRequest(): boolean;
    protected abstract doFinalGetResourceRequest(): Promise<void>;
    getMostRecentResponse(): HttpOperationResponse;
    getOperationResponse(): Promise<HttpOperationResponse>;
    getRestError(): RestError;
    protected updateState(url: string, shouldDeserialize: boolean | ((response: HttpOperationResponse) => boolean)): Promise<void>;
    /**
     * Retrieves operation status by querying the operation URL.
     * @param {string} statusUrl URL used to poll operation result.
     */
    protected updateOperationStatus(statusUrl: string, shouldDeserialize: boolean | ((response: HttpOperationResponse) => boolean)): Promise<HttpOperationResponse>;
    getPollState(): LROPollState;
}
export declare function getDelayInSeconds(azureServiceClient: AzureServiceClient, previousResponse: HttpOperationResponse): number;
/**
 * Get whether or not a long-running operation with the provided status is finished.
 * @param status The current status of a long-running operation.
 * @returns Whether or not a long-running operation with the provided status is finished.
 */
export declare function isFinished(status: LongRunningOperationStates): boolean;
export declare function longRunningOperationStatesEqual(lhs: LongRunningOperationStates, rhs: LongRunningOperationStates): boolean;
/**
 * Create a new long-running operation polling strategy based on the provided initial response.
 * @param initialResponse The initial response to the long-running operation's initial request.
 * @param azureServiceClient The AzureServiceClient that was used to send the initial request.
 * @param options Any options that were provided to the initial request.
 */
export declare function createLROPollStrategyFromInitialResponse(initialResponse: HttpOperationResponse, azureServiceClient: AzureServiceClient, options?: RequestOptionsBase): LROPollStrategy | undefined;
export declare function createLROPollStrategyFromPollState(azureServiceClient: AzureServiceClient, lroPollState: LROPollState): LROPollStrategy | undefined;
//# sourceMappingURL=lroPollStrategy.d.ts.map