// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License. See License.txt in the project root for license information.
import { __extends } from "tslib";
import { ServiceClient, getDefaultUserAgentValue as getDefaultUserAgentValueFromMsRest } from "@azure/ms-rest-js";
import { createLROPollerFromInitialResponse, createLROPollerFromPollState } from "./lroPoller";
import * as Constants from "./util/constants";
/**
 * @class
 * Initializes a new instance of the AzureServiceClient class.
 * @constructor
 *
 * @param {msRest.ServiceClientCredentilas} credentials - ApplicationTokenCredentials or
 * UserTokenCredentials object used for authentication.
 * @param {AzureServiceClientOptions} options - The parameter options used by AzureServiceClient
 */
var AzureServiceClient = /** @class */ (function (_super) {
    __extends(AzureServiceClient, _super);
    function AzureServiceClient(credentials, options) {
        var _a;
        var _this = _super.call(this, credentials, options = updateOptionsWithDefaultValues(options)) || this;
        _this.acceptLanguage = Constants.DEFAULT_LANGUAGE;
        // For convenience, if the credentials have an associated AzureEnvironment,
        // automatically use the baseUri from that environment.
        var env = (_a = credentials) === null || _a === void 0 ? void 0 : _a.environment;
        if (env && !_this.baseUri) {
            _this.baseUri = env.resourceManagerEndpointUrl;
        }
        if (options.acceptLanguage != undefined) {
            _this.acceptLanguage = options.acceptLanguage;
        }
        if (options.longRunningOperationRetryTimeout != undefined) {
            _this.longRunningOperationRetryTimeout = options.longRunningOperationRetryTimeout;
        }
        return _this;
    }
    /**
     * Send the initial request of a LRO (long running operation) and get back an
     * LROPoller that provides methods for polling the LRO and checking if the LRO is finished.
     * @param operationArguments The arguments to the operation.
     * @param operationSpec The specification for the operation.
     * @param options Additional options to be sent while making the request.
     * @returns The LROPoller object that provides methods for interacting with the LRO.
     */
    AzureServiceClient.prototype.sendLRORequest = function (operationArguments, operationSpec, options) {
        var _this = this;
        return this.sendOperationRequest(operationArguments, operationSpec)
            .then(function (initialResponse) { return createLROPollerFromInitialResponse(_this, initialResponse._response, options); });
    };
    /**
     * Provides a mechanism to make a request that will poll and provide the final result.
     * @param {msRest.RequestPrepareOptions|msRest.WebResource} request - The request object
     * @param {AzureRequestOptionsBase} [options] Additional options to be sent while making the request
     * @returns {Promise<msRest.HttpOperationResponse>} The HttpOperationResponse containing the final polling request, response and the responseBody.
     */
    AzureServiceClient.prototype.sendLongRunningRequest = function (request, options) {
        return this.beginLongRunningRequest(request, options)
            .then(function (lroResponse) { return lroResponse.pollUntilFinished(); })
            .then(function (res) { return res._response; });
    };
    /**
     * Send the initial request of a LRO (long running operation) and get back an
     * HttpLongRunningOperationResponse that provides methods for polling the LRO and checking if the
     * LRO is finished.
     * @param {msRest.RequestPrepareOptions|msRest.WebResource} request - The request object
     * @param {AzureRequestOptionsBase} [options] Additional options to be sent while making the request
     * @returns {Promise<LROPoller>} The HttpLongRunningOperationResponse
     * that provides methods for interacting with the LRO.
     */
    AzureServiceClient.prototype.beginLongRunningRequest = function (request, options) {
        var _this = this;
        return this.sendRequest(request)
            .then(function (initialResponse) { return createLROPollerFromInitialResponse(_this, initialResponse, options); });
    };
    /**
     * Restore an LROPoller from the provided LROPollState. This method can be used to recreate an
     * LROPoller on a different process or machine.
     */
    AzureServiceClient.prototype.restoreLROPoller = function (lroPollState) {
        return createLROPollerFromPollState(this, lroPollState);
    };
    return AzureServiceClient;
}(ServiceClient));
export { AzureServiceClient };
export function getDefaultUserAgentValue() {
    var defaultUserAgent = getDefaultUserAgentValueFromMsRest();
    return "ms-rest-azure-js/" + Constants.msRestAzureVersion + " " + defaultUserAgent;
}
export function updateOptionsWithDefaultValues(options) {
    if (!options) {
        options = {};
    }
    if (options.generateClientRequestIdHeader == undefined) {
        options.generateClientRequestIdHeader = true;
    }
    if (!options.userAgent) {
        options.userAgent = getDefaultUserAgentValue();
    }
    return options;
}
//# sourceMappingURL=azureServiceClient.js.map