"""
CLion CMake quick documentation builder

This is a Sphinx extension, used to generate Cmake quick doc.
The main purpose of this builder is to convert links, so that they are easier
to process in `CMakeDocumentationLinkHandler`.

See also conf.py
"""

from typing import Any, Dict
from urllib.parse import quote, urlparse

from sphinx.application import Sphinx
from sphinx.builders.html import StandaloneHTMLBuilder
from sphinx.util.console import darkgreen  # type: ignore


class QuickDocBuilder(StandaloneHTMLBuilder):
    """
    This builder is used in CLion to generate quick documentation pages.
    """
    name = 'quickdoc'

    def get_relative_uri(self, from_: str, to: str, typ: str = None) -> str:
        """
        Always use absolute path for quick doc

        This way it's much easier to interpret references inside quick doc.
        """
        return self.get_target_uri(to, typ)

    def get_target_uri(self, docname: str, typ: str = None) -> str:
        """
        Convert internal URL to external ones

        CLion only bundles a subset of the whole CMake documentation according to
        mapped_dirs. Those URLs pointing to the bundled docs, should use
        relative URLs so that they can be resolved internally in CLion.
        However, those pointing to non-bundled docs, should open external browser.
        Those urls should be external URL pointing to CMake website (see external_url).
        """
        external_url = self.env.config["cmake_external_url"]
        mapped_dirs = self.env.config["cmake_mapped_dirs"]
        doc_category = urlparse(docname).path.split("/")[0]
        if doc_category in mapped_dirs:
            return super().get_target_uri(docname, typ)
        else:
            return external_url + quote(docname) + self.link_suffix


def setup(app: Sphinx) -> Dict[str, Any]:
    app.add_config_value("cmake_external_url", '', 'env')
    app.add_config_value("cmake_mapped_dirs", [], 'env')

    app.add_builder(QuickDocBuilder)

    return {
      'version': '1.0',
      'parallel_read_safe': True,
      'parallel_write_safe': True,
    }