/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.azurebfs.AbfsStatistic;
import org.apache.hadoop.fs.azurebfs.AzureBlobFileSystemStore;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.TimeoutException;
import org.apache.hadoop.fs.azurebfs.contracts.services.AzureServiceErrorCode;
import org.apache.hadoop.fs.azurebfs.enums.BlobCopyProgress;
import org.apache.hadoop.fs.azurebfs.services.AbfsBlobClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsLease;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperation;
import org.apache.hadoop.fs.azurebfs.services.ListActionTaker;
import org.apache.hadoop.fs.azurebfs.services.PathInformation;
import org.apache.hadoop.fs.azurebfs.services.RenameAtomicity;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlobRenameHandler
extends ListActionTaker {
    public static final Logger LOG = LoggerFactory.getLogger(AbfsClient.class);
    private final String srcEtag;
    private final Path src;
    private final Path dst;
    private final boolean isAtomicRename;
    private final boolean isAtomicRenameRecovery;
    private final TracingContext tracingContext;
    private AbfsLease srcAbfsLease;
    private String srcLeaseId;
    private final List<AbfsLease> leases = new ArrayList<AbfsLease>();
    private final AtomicInteger operatedBlobCount = new AtomicInteger(0);

    public BlobRenameHandler(String src, String dst, AbfsBlobClient abfsClient, String srcEtag, boolean isAtomicRename, boolean isAtomicRenameRecovery, TracingContext tracingContext) {
        super(new Path(src), abfsClient, tracingContext);
        this.srcEtag = srcEtag;
        this.tracingContext = tracingContext;
        this.src = new Path(src);
        this.dst = new Path(dst);
        this.isAtomicRename = isAtomicRename;
        this.isAtomicRenameRecovery = isAtomicRenameRecovery;
    }

    @Override
    int getMaxConsumptionParallelism() {
        return this.getAbfsClient().getAbfsConfiguration().getBlobRenameDirConsumptionParallelism();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean execute(boolean isRenameRecovery) throws AzureBlobFileSystemException {
        PathInformation pathInformation = this.getPathInformation(this.src, this.tracingContext);
        boolean result = false;
        if (this.preCheck(this.src, this.dst, pathInformation, isRenameRecovery)) {
            RenameAtomicity renameAtomicity = null;
            if (pathInformation.getIsDirectory().booleanValue() && pathInformation.getIsImplicit().booleanValue()) {
                try {
                    AbfsRestOperation createMarkerOp = this.getAbfsClient().createMarkerAtPath(this.src.toUri().getPath(), null, null, this.tracingContext);
                    pathInformation.setETag(AzureBlobFileSystemStore.extractEtagHeader(createMarkerOp.getResult()));
                }
                catch (AbfsRestOperationException ex) {
                    LOG.debug("Marker creation failed for src path {} ", (Object)this.src.toUri().getPath());
                }
            }
            try {
                if (this.isAtomicRename) {
                    this.getAbfsClient().getAbfsCounters().incrementCounter(AbfsStatistic.ATOMIC_RENAME_PATH_ATTEMPTS, 1L);
                    if (this.srcAbfsLease == null) {
                        this.srcAbfsLease = this.takeLease(this.src, this.srcEtag);
                    }
                    this.srcLeaseId = this.srcAbfsLease.getLeaseID();
                    if (!this.isAtomicRenameRecovery && pathInformation.getIsDirectory().booleanValue()) {
                        renameAtomicity = this.getRenameAtomicity(pathInformation);
                        renameAtomicity.preRename();
                    }
                }
                result = pathInformation.getIsDirectory().booleanValue() ? this.listRecursiveAndTakeAction() && this.finalSrcRename() : this.renameInternal(this.src, this.dst);
            }
            finally {
                if (this.srcAbfsLease != null) {
                    if (result) {
                        this.srcAbfsLease.cancelTimer();
                    } else {
                        this.srcAbfsLease.free();
                    }
                }
            }
            if (result && renameAtomicity != null) {
                renameAtomicity.postRename();
            }
        }
        return result;
    }

    private boolean finalSrcRename() throws AzureBlobFileSystemException {
        this.tracingContext.setOperatedBlobCount(this.operatedBlobCount.get() + 1);
        try {
            boolean bl = this.renameInternal(this.src, this.dst);
            return bl;
        }
        catch (AbfsRestOperationException e) {
            if (e.getStatusCode() == 409) {
                this.getAbfsClient().deleteBlobPath(this.src, null, this.tracingContext);
                boolean bl = true;
                return bl;
            }
            throw e;
        }
        finally {
            this.tracingContext.setOperatedBlobCount(null);
        }
    }

    @VisibleForTesting
    public RenameAtomicity getRenameAtomicity(PathInformation pathInformation) {
        return new RenameAtomicity(this.src, this.dst, new Path(this.src.getParent(), this.src.getName() + "-RenamePending.json"), this.tracingContext, pathInformation.getETag(), this.getAbfsClient());
    }

    private AbfsLease takeLease(Path path, String eTag) throws AzureBlobFileSystemException {
        AbfsLease lease = new AbfsLease(this.getAbfsClient(), path.toUri().getPath(), false, this.getAbfsClient().getAbfsConfiguration().getAtomicRenameLeaseRefreshDuration(), eTag, this.tracingContext);
        this.leases.add(lease);
        return lease;
    }

    private boolean containsColon(Path p) {
        return p.toUri().getPath().contains(":");
    }

    private boolean preCheck(Path src, Path dst, PathInformation pathInformation, boolean isRenameRecovery) throws AzureBlobFileSystemException {
        this.validateDestinationIsNotSubDir(src, dst);
        this.validateSourcePath(pathInformation);
        this.validateDestinationPathNotExist(src, dst, pathInformation, isRenameRecovery);
        this.validateDestinationParentExist(src, dst, pathInformation);
        return true;
    }

    private void validateDestinationIsNotSubDir(Path src, Path dst) throws AbfsRestOperationException {
        LOG.debug("Check if the destination is subDirectory");
        Path nestedDstParent = dst.getParent();
        if (nestedDstParent != null && nestedDstParent.toUri().getPath().indexOf(src.toUri().getPath()) == 0) {
            LOG.info("Rename src: {} dst: {} failed as dst is subDir of src", (Object)src, (Object)dst);
            throw new AbfsRestOperationException(409, AzureServiceErrorCode.INVALID_RENAME_SOURCE_PATH.getErrorCode(), AzureServiceErrorCode.INVALID_RENAME_SOURCE_PATH.getErrorMessage(), new Exception(AzureServiceErrorCode.INVALID_RENAME_SOURCE_PATH.getErrorCode()));
        }
    }

    private void validateSourcePath(PathInformation pathInformation) throws AzureBlobFileSystemException {
        if (!pathInformation.getPathExists().booleanValue()) {
            throw new AbfsRestOperationException(404, AzureServiceErrorCode.SOURCE_PATH_NOT_FOUND.getErrorCode(), null, new Exception(AzureServiceErrorCode.SOURCE_PATH_NOT_FOUND.getErrorCode()));
        }
        if (this.srcEtag != null && !this.srcEtag.equals(pathInformation.getETag())) {
            throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_ALREADY_EXISTS.getErrorCode(), null, new Exception(AzureServiceErrorCode.PATH_ALREADY_EXISTS.getErrorCode()));
        }
    }

    private void validateDestinationPathNotExist(Path src, Path dst, PathInformation pathInformation, boolean isRenameRecovery) throws AzureBlobFileSystemException {
        PathInformation dstPathInformation;
        if (!isRenameRecovery && pathInformation.getIsDirectory().booleanValue() && dst.getName().equals(src.getName()) && (dstPathInformation = this.getPathInformation(dst, this.tracingContext)).getPathExists().booleanValue()) {
            LOG.info("Rename src: {} dst: {} failed as qualifiedDst already exists", (Object)src, (Object)dst);
            throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_ALREADY_EXISTS.getErrorCode(), null, null);
        }
    }

    private void validateDestinationParentExist(Path src, Path dst, PathInformation pathInformation) throws AzureBlobFileSystemException {
        PathInformation nestedDstInfo;
        Path nestedDstParent = dst.getParent();
        if (!(dst.isRoot() || nestedDstParent == null || nestedDstParent.isRoot() || pathInformation.getIsDirectory().booleanValue() && dst.getName().equals(src.getName()) || (nestedDstInfo = this.getPathInformation(nestedDstParent, this.tracingContext)).getPathExists().booleanValue() && nestedDstInfo.getIsDirectory().booleanValue())) {
            throw new AbfsRestOperationException(404, AzureServiceErrorCode.RENAME_DESTINATION_PARENT_PATH_NOT_FOUND.getErrorCode(), null, new Exception(AzureServiceErrorCode.RENAME_DESTINATION_PARENT_PATH_NOT_FOUND.getErrorCode()));
        }
    }

    @Override
    boolean takeAction(Path path) throws AzureBlobFileSystemException {
        return this.renameInternal(path, this.getDstPathForBlob(this.dst, path, this.src));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean renameInternal(Path path, Path destinationPathForBlobPartOfRenameSrcDir) throws AzureBlobFileSystemException {
        String leaseId;
        AbfsLease abfsLease = null;
        if (this.isAtomicRename) {
            if (path.equals((Object)this.src)) {
                abfsLease = this.srcAbfsLease;
                leaseId = this.srcLeaseId;
            } else {
                abfsLease = this.takeLease(path, null);
                leaseId = abfsLease.getLeaseID();
            }
        } else {
            leaseId = null;
        }
        boolean operated = false;
        try {
            this.copyPath(path, destinationPathForBlobPartOfRenameSrcDir, leaseId);
            this.getAbfsClient().deleteBlobPath(path, leaseId, this.tracingContext);
            operated = true;
        }
        finally {
            if (abfsLease != null) {
                if (operated) {
                    abfsLease.cancelTimer();
                } else {
                    abfsLease.free();
                }
            }
        }
        this.operatedBlobCount.incrementAndGet();
        return true;
    }

    private void copyPath(Path src, Path dst, String leaseId) throws AzureBlobFileSystemException {
        String copyId;
        try {
            AbfsRestOperation copyPathOp = this.getAbfsClient().copyBlob(src, dst, leaseId, this.tracingContext);
            String progress = copyPathOp.getResult().getResponseHeader("x-ms-copy-status");
            if ("success".equalsIgnoreCase(progress)) {
                return;
            }
            copyId = copyPathOp.getResult().getResponseHeader("x-ms-copy-id");
        }
        catch (AbfsRestOperationException ex) {
            if (ex.getStatusCode() == 409) {
                AbfsRestOperation dstPathStatus = this.getAbfsClient().getPathStatus(dst.toUri().getPath(), this.tracingContext, null, false);
                String srcCopyPath = "/" + this.getAbfsClient().getFileSystem() + src.toUri().getPath();
                if (dstPathStatus != null && dstPathStatus.getResult() != null && srcCopyPath.equals(this.getDstSource(dstPathStatus))) {
                    return;
                }
            }
            throw ex;
        }
        long pollWait = this.getAbfsClient().getAbfsConfiguration().getBlobCopyProgressPollWaitMillis();
        long maxWait = this.getAbfsClient().getAbfsConfiguration().getBlobCopyProgressMaxWaitMillis();
        long startTime = System.currentTimeMillis();
        while (this.handleCopyInProgress(dst, this.tracingContext, copyId) == BlobCopyProgress.PENDING) {
            if (System.currentTimeMillis() - startTime > maxWait) {
                throw new TimeoutException(String.format("Blob copy progress wait time exceeded for source: %s and destination: %s", src, dst));
            }
            try {
                Thread.sleep(pollWait);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    private String getDstSource(AbfsRestOperation dstPathStatus) {
        try {
            String responseHeader = dstPathStatus.getResult().getResponseHeader("x-ms-copy-source");
            if (responseHeader == null) {
                return null;
            }
            return new URL(responseHeader).toURI().getPath();
        }
        catch (MalformedURLException | URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    public BlobCopyProgress handleCopyInProgress(Path dstPath, TracingContext tracingContext, String copyId) throws AzureBlobFileSystemException {
        AbfsRestOperation op = this.getAbfsClient().getPathStatus(dstPath.toUri().getPath(), tracingContext, null, false);
        if (op.getResult() != null && copyId != null && copyId.equals(op.getResult().getResponseHeader("x-ms-copy-id"))) {
            String copyStatus = op.getResult().getResponseHeader("x-ms-copy-status");
            if ("success".equalsIgnoreCase(copyStatus)) {
                return BlobCopyProgress.SUCCESS;
            }
            if ("failed".equalsIgnoreCase(copyStatus)) {
                throw new AbfsRestOperationException(AzureServiceErrorCode.COPY_BLOB_FAILED.getStatusCode(), AzureServiceErrorCode.COPY_BLOB_FAILED.getErrorCode(), String.format("copy to path %s failed due to: %s", dstPath.toUri().getPath(), op.getResult().getResponseHeader("x-ms-copy-status-description")), new Exception(AzureServiceErrorCode.COPY_BLOB_FAILED.getErrorCode()));
            }
            if ("aborted".equalsIgnoreCase(copyStatus)) {
                throw new AbfsRestOperationException(AzureServiceErrorCode.COPY_BLOB_ABORTED.getStatusCode(), AzureServiceErrorCode.COPY_BLOB_ABORTED.getErrorCode(), String.format("copy to path %s aborted", dstPath.toUri().getPath()), new Exception(AzureServiceErrorCode.COPY_BLOB_ABORTED.getErrorCode()));
            }
        }
        return BlobCopyProgress.PENDING;
    }

    private Path getDstPathForBlob(Path destinationDir, Path blobPath, Path sourceDir) {
        String srcBlobPropertyPathStr;
        String destinationPathStr = destinationDir.toUri().getPath();
        String sourcePathStr = sourceDir.toUri().getPath();
        if (sourcePathStr.equals(srcBlobPropertyPathStr = blobPath.toUri().getPath())) {
            return destinationDir;
        }
        return new Path(destinationPathStr + "/" + srcBlobPropertyPathStr.substring(sourcePathStr.length()));
    }

    private PathInformation getPathInformation(Path path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        try {
            AbfsRestOperation op = this.getAbfsClient().getPathStatus(path.toString(), tracingContext, null, true);
            return new PathInformation(true, this.getAbfsClient().checkIsDir(op.getResult()), AzureBlobFileSystemStore.extractEtagHeader(op.getResult()), op.getResult() instanceof AbfsHttpOperation.AbfsHttpOperationWithFixedResultForGetFileStatus);
        }
        catch (AbfsRestOperationException e) {
            if (e.getStatusCode() == 404) {
                return new PathInformation(false, false, null, false);
            }
            throw e;
        }
    }

    @VisibleForTesting
    public List<AbfsLease> getLeases() {
        return this.leases;
    }
}

