"use strict";

var _child_process = require("child_process");
var _crypto = require("crypto");
var _dotenv = _interopRequireDefault(require("dotenv"));
var _electron = require("electron");
var _findProcess = _interopRequireDefault(require("find-process"));
var fsPromises = _interopRequireWildcard(require("fs/promises"));
var _nodeFs = _interopRequireDefault(require("node:fs"));
var _nodeOs = require("node:os");
var _nodeUtil = require("node:util");
var _os = require("os");
var _path = _interopRequireDefault(require("path"));
var _url = _interopRequireDefault(require("url"));
var _yargs = _interopRequireDefault(require("yargs"));
var _helpers = require("yargs/helpers");
var _i18next = _interopRequireDefault(require("./i18next.config"));
var _pluginManagement = require("./plugin-management");
var _windowSize = _interopRequireDefault(require("./windowSize"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
_dotenv.default.config({
  path: _path.default.join(process.resourcesPath, '.env')
});
const pathInfoDebug = false;
let pathInfo;
const isDev = process.env.ELECTRON_DEV || false;
let frontendPath = '';
if (isDev) {
  frontendPath = _path.default.resolve('..', 'frontend', 'build', 'index.html');
} else {
  frontendPath = _path.default.join(process.resourcesPath, 'frontend', 'index.html');
}
const backendToken = (0, _crypto.randomBytes)(32).toString('hex');
const startUrl = (process.env.ELECTRON_START_URL || _url.default.format({
  pathname: frontendPath,
  protocol: 'file:',
  slashes: true
})

// Windows paths use backslashes and for consistency we want to use forward slashes.
// For example: when application triggers refresh it requests a URL with forward slashes and
// we use startUrl to determine if it's an internal or external URL. So it's easier to
// convert everything to forward slashes.
).replace(/\\/g, '/');
const args = (0, _yargs.default)((0, _helpers.hideBin)(process.argv)).command('list-plugins', 'List all static and user-added plugins.', () => {}, () => {
  try {
    const backendPath = _path.default.join(process.resourcesPath, 'headlamp-server');
    const stdout = (0, _child_process.execSync)(`${backendPath} list-plugins`);
    process.stdout.write(stdout);
    process.exit(0);
  } catch (error) {
    console.error(`Error listing plugins: ${error}`);
    process.exit(1);
  }
}).options({
  headless: {
    describe: 'Open Headlamp in the default web browser instead of its app window',
    type: 'boolean'
  },
  'disable-gpu': {
    describe: 'Disable use of GPU. For people who may have buggy graphics drivers',
    type: 'boolean'
  }
}).positional('kubeconfig', {
  describe: 'Path to the kube config file (uses the default kube config location if not specified)',
  type: 'string'
}).help().parseSync();
const isHeadlessMode = args.headless === true;
let disableGPU = args['disable-gpu'] === true;
const defaultPort = 4466;
const useExternalServer = process.env.EXTERNAL_SERVER || false;
const shouldCheckForUpdates = process.env.HEADLAMP_CHECK_FOR_UPDATES !== 'false';
const manifestDir = isDev ? _path.default.resolve('./') : process.resourcesPath;
const manifestFile = _path.default.join(manifestDir, 'app-build-manifest.json');
const buildManifest = _nodeFs.default.existsSync(manifestFile) ? require(manifestFile) : {};

// make it global so that it doesn't get garbage collected
let mainWindow;

/**
 * `Action` is an interface for an action to be performed by the plugin manager.
 *
 * @interface
 * @property {string} identifier - The unique identifier for the action.
 * @property {'INSTALL' | 'UNINSTALL' | 'UPDATE' | 'LIST' | 'CANCEL' | 'GET'} action - The type of the action.
 * @property {string} [URL] - The URL for the action. Optional.
 * @property {string} [destinationFolder] - The destination folder for the action. Optional.
 * @property {string} [headlampVersion] - The version of Headlamp for the action. Optional.
 * @property {string} [pluginName] - The name of the plugin for the action. Optional.
 */

/**
 * `ProgressResp` is an interface for progress response.
 *
 * @interface
 * @property {string} type - The type of the progress response.
 * @property {string} message - The message of the progress response.
 * @property {Record<string, any>} data - Additional data for the progress response. Optional.
 */

/**
 * `PluginManagerEventListeners` is a class that manages event listeners for plugins-manager.
 *
 * @class
 */
class PluginManagerEventListeners {
  cache = {};
  constructor() {
    this.cache = {};
  }

  /**
   * Converts the progress response to a percentage.
   *
   * @param {ProgressResp} progress - The progress response object.
   * @returns {number} The progress as a percentage.
   */
  convertProgressToPercentage(progress) {
    switch (progress.message) {
      case 'Fetching Plugin Metadata':
        return 20;
      case 'Plugin Metadata Fetched':
        return 30;
      case 'Downloading Plugin':
        return 50;
      case 'Plugin Downloaded':
        return 100;
      default:
        return 0;
    }
  }

  /**
   * Sets up event handlers for plugin-manager.
   *
   * @method
   * @name setupEventHandlers
   */
  setupEventHandlers() {
    _electron.ipcMain.on('plugin-manager', async (event, data) => {
      const eventData = JSON.parse(data);
      const {
        identifier,
        action
      } = eventData;
      const updateCache = progress => {
        const percentage = this.convertProgressToPercentage(progress);
        this.cache[identifier].progress = progress;
        this.cache[identifier].percentage = percentage;
      };
      switch (action) {
        case 'INSTALL':
          this.handleInstall(eventData, updateCache);
          break;
        case 'UPDATE':
          this.handleUpdate(eventData, updateCache);
          break;
        case 'UNINSTALL':
          this.handleUninstall(eventData, updateCache);
          break;
        case 'LIST':
          this.handleList(event, eventData);
          break;
        case 'CANCEL':
          this.handleCancel(event, identifier);
          break;
        case 'GET':
          this.handleGet(event, identifier);
          break;
        default:
          console.error(`Unknown action: ${action}`);
      }
    });
  }

  /**
   * Handles the installation process.
   *
   * @method
   * @name handleInstall
   * @private
   */
  async handleInstall(eventData, updateCache) {
    const {
      identifier,
      URL,
      destinationFolder,
      headlampVersion,
      pluginName
    } = eventData;
    if (!mainWindow) {
      return {
        type: 'error',
        message: 'Main window is not available'
      };
    }
    if (!URL) {
      return {
        type: 'error',
        message: 'URL is required'
      };
    }
    const controller = new AbortController();
    this.cache[identifier] = {
      action: 'INSTALL',
      progress: {
        type: 'info',
        message: 'waiting for user consent'
      },
      percentage: 0,
      controller
    };
    let pluginInfo;
    try {
      pluginInfo = await _pluginManagement.PluginManager.fetchPluginInfo(URL, {
        signal: controller.signal
      });
    } catch (error) {
      console.error('Error fetching plugin info:', error);
      _electron.dialog.showErrorBox(_i18next.default.t('Failed to fetch plugin info'), _i18next.default.t('An error occurred while fetching plugin info from {{  URL }}.', {
        URL
      }));
      return {
        type: 'error',
        message: 'Failed to fetch plugin info'
      };
    }
    const dialogOptions = {
      type: 'question',
      buttons: ['Yes', 'No'],
      defaultId: 1,
      title: 'Plugin Installation',
      message: 'Do you want to install this plugin?',
      detail: `You are about to install ${pluginName} plugin from: ${pluginInfo.archiveURL}\nDo you want to proceed?`
    };
    let userChoice;
    try {
      const answer = await _electron.dialog.showMessageBox(mainWindow, dialogOptions);
      userChoice = answer.response;
    } catch (error) {
      console.error('Error during installation process:', error);
      return {
        type: 'error',
        message: 'An error occurred during the installation process'
      };
    }
    console.log('User response:', userChoice);
    if (userChoice === 1) {
      // User clicked "No"
      this.cache[identifier] = {
        action: 'INSTALL',
        progress: {
          type: 'error',
          message: 'installation cancelled due to user consent'
        },
        percentage: 0,
        controller
      };
      return {
        type: 'error',
        message: 'Installation cancelled due to user consent'
      };
    }

    // User clicked "Yes", proceed with installation
    this.cache[identifier] = {
      action: 'INSTALL',
      progress: {
        type: 'info',
        message: 'installing plugin'
      },
      percentage: 10,
      controller
    };
    _pluginManagement.PluginManager.installFromPluginPkg(pluginInfo, destinationFolder, headlampVersion, progress => {
      updateCache(progress);
    }, controller.signal);
    return {
      type: 'info',
      message: 'Installation started'
    };
  }
  /**
   * Handles the update process.
   *
   * @method
   * @name handleUpdate
   * @private
   */
  handleUpdate(eventData, updateCache) {
    const {
      identifier,
      pluginName,
      destinationFolder,
      headlampVersion
    } = eventData;
    if (!pluginName) {
      this.cache[identifier] = {
        action: 'UPDATE',
        progress: {
          type: 'error',
          message: 'Plugin Name is required'
        }
      };
      return;
    }
    const controller = new AbortController();
    this.cache[identifier] = {
      action: 'UPDATE',
      percentage: 10,
      progress: {
        type: 'info',
        message: 'updating plugin'
      },
      controller
    };
    _pluginManagement.PluginManager.update(pluginName, destinationFolder, headlampVersion, progress => {
      updateCache(progress);
    }, controller.signal);
  }

  /**
   * Handles the uninstallation process.
   *
   * @method
   * @name handleUninstall
   * @private
   */
  handleUninstall(eventData, updateCache) {
    const {
      identifier,
      pluginName,
      destinationFolder
    } = eventData;
    if (!pluginName) {
      this.cache[identifier] = {
        action: 'UNINSTALL',
        progress: {
          type: 'error',
          message: 'Plugin Name is required'
        }
      };
      return;
    }
    this.cache[identifier] = {
      action: 'UNINSTALL',
      progress: {
        type: 'info',
        message: 'uninstalling plugin'
      }
    };
    _pluginManagement.PluginManager.uninstall(pluginName, destinationFolder, progress => {
      updateCache(progress);
    });
  }

  /**
   * Handles the list event.
   *
   * @method
   * @name handleList
   * @param {Electron.IpcMainEvent} event - The IPC Main Event.
   * @param {Action} eventData - The event data.
   * @private
   */
  handleList(event, eventData) {
    const {
      identifier,
      destinationFolder
    } = eventData;
    _pluginManagement.PluginManager.list(destinationFolder, progress => {
      event.sender.send('plugin-manager', JSON.stringify({
        identifier: identifier,
        ...progress
      }));
    });
  }

  /**
   * Handles the cancel event.
   *
   * @method
   * @name handleCancel
   * @param {Electron.IpcMainEvent} event - The IPC Main Event.
   * @param {string} identifier - The identifier of the event to cancel.
   * @private
   */
  handleCancel(event, identifier) {
    const cacheEntry = this.cache[identifier];
    if (cacheEntry?.controller) {
      cacheEntry.controller.abort();
      event.sender.send('plugin-manager', JSON.stringify({
        type: 'success',
        message: 'cancelled'
      }));
    }
  }

  /**
   * Handles the get event.
   *
   * @method
   * @name handleGet
   * @param {Electron.IpcMainEvent} event - The IPC Main Event.
   * @param {string} identifier - The identifier of the event to get.
   * @private
   */
  handleGet(event, identifier) {
    const cacheEntry = this.cache[identifier];
    if (cacheEntry) {
      event.sender.send('plugin-manager', JSON.stringify({
        identifier: identifier,
        ...cacheEntry.progress,
        percentage: cacheEntry.percentage
      }));
    } else {
      event.sender.send('plugin-manager', JSON.stringify({
        type: 'error',
        message: 'No such operation in progress'
      }));
    }
  }
}

/**
 * Returns the user's preferred shell or a fallback shell.
 * @returns A promise that resolves to the shell path.
 */
async function getShell() {
  // Fallback chain
  const shells = ['/bin/zsh', '/bin/bash', '/bin/sh'];
  let userShell = '';
  try {
    userShell = (0, _nodeOs.userInfo)().shell || process.env.SHELL || '';
    if (userShell) shells.unshift(userShell);
  } catch (error) {
    console.error('Failed to get user shell:', error);
  }
  for (const shell of shells) {
    try {
      await fsPromises.stat(shell);
      return shell;
    } catch (error) {
      console.error(`Shell not found: ${shell}, error: ${error}`);
    }
  }
  console.error('No valid shell found, defaulting to /bin/sh');
  return '/bin/sh';
}

/**
 * Retrieves the environment variables from the user's shell.
 * @returns A promise that resolves to the shell environment.
 */
async function getShellEnv() {
  const execPromisify = (0, _nodeUtil.promisify)(_child_process.exec);
  const shell = await getShell();
  const isWindows = process.platform === 'win32';

  // For Windows, just return the current environment
  if (isWindows) {
    return {
      ...process.env
    };
  }

  // For Unix-like systems
  const isZsh = shell.includes('zsh');
  // interactive is supported only on zsh
  const shellArgs = isZsh ? ['--login', '--interactive', '-c'] : ['--login', '-c'];
  try {
    const env = {
      ...process.env,
      DISABLE_AUTO_UPDATE: 'true'
    };
    let stdout;
    let isEnvNull = false;
    try {
      // Try env -0 first
      const command = 'env -0';
      ({
        stdout
      } = await execPromisify(`${shell} ${shellArgs.join(' ')} '${command}'`, {
        encoding: 'utf8',
        timeout: 10000,
        env
      }));
      isEnvNull = true;
    } catch (error) {
      // If env -0 fails, fall back to env
      console.log('env -0 failed, falling back to env');
      const command = 'env';
      ({
        stdout
      } = await execPromisify(`${shell} ${shellArgs.join(' ')} '${command}'`, {
        encoding: 'utf8',
        timeout: 10000,
        env
      }));
    }
    const processLines = separator => {
      return stdout.split(separator).reduce((acc, line) => {
        const firstEqualIndex = line.indexOf('=');
        if (firstEqualIndex > 0) {
          const key = line.slice(0, firstEqualIndex);
          const value = line.slice(firstEqualIndex + 1);
          acc[key] = value;
        }
        return acc;
      }, {});
    };
    const envVars = isEnvNull ? processLines('\0') : processLines('\n');
    const mergedEnv = {
      ...process.env,
      ...envVars
    };
    return mergedEnv;
  } catch (error) {
    console.error('Failed to get shell environment:', error);
    return process.env;
  }
}
async function startServer(flags = []) {
  const serverFilePath = isDev ? _path.default.resolve('../backend/headlamp-server') : _path.default.join(process.resourcesPath, './headlamp-server');
  let serverArgs = [];
  if (!!args.kubeconfig) {
    serverArgs = serverArgs.concat(['--kubeconfig', args.kubeconfig]);
  }
  const proxyUrls = !!buildManifest && buildManifest['proxy-urls'];
  if (!!proxyUrls && proxyUrls.length > 0) {
    serverArgs = serverArgs.concat(['--proxy-urls', proxyUrls.join(',')]);
  }
  const bundledPlugins = _path.default.join(process.resourcesPath, '.plugins');
  function isEmpty(path) {
    return _nodeFs.default.readdirSync(path).length === 0;
  }

  // Enable the Helm and dynamic cluster endpoints
  process.env.HEADLAMP_CONFIG_ENABLE_HELM = 'true';
  process.env.HEADLAMP_CONFIG_ENABLE_DYNAMIC_CLUSTERS = 'true';

  // Pass a token to the backend that can be used for auth on some routes
  process.env.HEADLAMP_BACKEND_TOKEN = backendToken;

  // Set the bundled plugins in addition to the the user's plugins.
  if (_nodeFs.default.existsSync(bundledPlugins) && !isEmpty(bundledPlugins)) {
    process.env.HEADLAMP_STATIC_PLUGINS_DIR = bundledPlugins;
  }
  serverArgs = serverArgs.concat(flags);
  console.log('arguments passed to backend server', serverArgs);
  let extendedEnv;
  try {
    extendedEnv = await getShellEnv();
  } catch (error) {
    console.error('Failed to get shell environment, using default:', error);
    extendedEnv = process.env;
  }
  const options = {
    detached: true,
    windowsHide: true,
    env: {
      ...extendedEnv
    }
  };
  return (0, _child_process.spawn)(serverFilePath, serverArgs, options);
}

/**
 * Are we running inside WSL?
 * @returns true if we are running inside WSL.
 */
function isWSL() {
  try {
    const data = _nodeFs.default.readFileSync('/proc/version', {
      encoding: 'utf8',
      flag: 'r'
    });
    return data.indexOf('icrosoft') !== -1;
  } catch {
    return false;
  }
}
let serverProcess;
let intentionalQuit;
let serverProcessQuit;
function quitServerProcess() {
  if ((!serverProcess || serverProcessQuit) && process.platform !== 'win32') {
    console.error('server process already not running');
    return;
  }
  intentionalQuit = true;
  console.info('stopping server process...');
  if (!serverProcess) {
    return;
  }
  serverProcess.stdin.destroy();
  // @todo: should we try and end the process a bit more gracefully?
  //       What happens if the kill signal doesn't kill it?
  serverProcess.kill();
  serverProcess = null;
}
function getAcceleratorForPlatform(navigation) {
  switch ((0, _os.platform)()) {
    case 'darwin':
      return navigation === 'right' ? 'Cmd+]' : 'Cmd+[';
    case 'win32':
      return navigation === 'right' ? 'Alt+Right' : 'Alt+Left';
    default:
      return navigation === 'right' ? 'Alt+Right' : 'Alt+Left';
  }
}
function getDefaultAppMenu() {
  const isMac = process.platform === 'darwin';
  const sep = {
    type: 'separator'
  };
  const aboutMenu = {
    label: _i18next.default.t('About'),
    role: 'about',
    id: 'original-about',
    afterPlugins: true
  };
  const quitMenu = {
    label: _i18next.default.t('Quit'),
    role: 'quit',
    id: 'original-quit'
  };
  const selectAllMenu = {
    label: _i18next.default.t('Select All'),
    role: 'selectAll',
    id: 'original-select-all'
  };
  const deleteMenu = {
    label: _i18next.default.t('Delete'),
    role: 'delete',
    id: 'original-delete'
  };
  const appMenu = [
  // { role: 'appMenu' }
  ...(isMac ? [{
    label: _electron.app.name,
    submenu: [aboutMenu, sep, {
      label: _i18next.default.t('Services'),
      role: 'services',
      id: 'original-services'
    }, sep, {
      label: _i18next.default.t('Hide'),
      role: 'hide',
      id: 'original-hide'
    }, {
      label: _i18next.default.t('Hide Others'),
      role: 'hideothers',
      id: 'original-hide-others'
    }, {
      label: _i18next.default.t('Show All'),
      role: 'unhide',
      id: 'original-show-all'
    }, sep, quitMenu]
  }] : []),
  // { role: 'fileMenu' }
  {
    label: _i18next.default.t('File'),
    id: 'original-file',
    submenu: [isMac ? {
      label: _i18next.default.t('Close'),
      role: 'close',
      id: 'original-close'
    } : quitMenu]
  },
  // { role: 'editMenu' }
  {
    label: _i18next.default.t('Edit'),
    id: 'original-edit',
    submenu: [{
      label: _i18next.default.t('Cut'),
      role: 'cut',
      id: 'original-cut'
    }, {
      label: _i18next.default.t('Copy'),
      role: 'copy',
      id: 'original-copy'
    }, {
      label: _i18next.default.t('Paste'),
      role: 'paste',
      id: 'original-paste'
    }, ...(isMac ? [{
      label: _i18next.default.t('Paste and Match Style'),
      role: 'pasteAndMatchStyle',
      id: 'original-paste-and-match-style'
    }, deleteMenu, selectAllMenu, sep, {
      label: _i18next.default.t('Speech'),
      id: 'original-speech',
      submenu: [{
        label: _i18next.default.t('Start Speaking'),
        role: 'startspeaking',
        id: 'original-start-speaking'
      }, {
        label: _i18next.default.t('Stop Speaking'),
        role: 'stopspeaking',
        id: 'original-stop-speaking'
      }]
    }] : [deleteMenu, sep, selectAllMenu])]
  },
  // { role: 'viewMenu' }
  {
    label: _i18next.default.t('View'),
    id: 'original-view',
    submenu: [{
      label: _i18next.default.t('Toggle Developer Tools'),
      role: 'toggledevtools',
      id: 'original-toggle-dev-tools'
    }, sep, {
      label: _i18next.default.t('Reset Zoom'),
      role: 'resetzoom',
      id: 'original-reset-zoom'
    }, {
      label: _i18next.default.t('Zoom In'),
      role: 'zoomin',
      id: 'original-zoom-in'
    }, {
      label: _i18next.default.t('Zoom Out'),
      role: 'zoomout',
      id: 'original-zoom-out'
    }, sep, {
      label: _i18next.default.t('Toggle Fullscreen'),
      role: 'togglefullscreen',
      id: 'original-toggle-fullscreen'
    }]
  }, {
    label: _i18next.default.t('Navigate'),
    id: 'original-navigate',
    submenu: [{
      label: _i18next.default.t('Reload'),
      role: 'forcereload',
      id: 'original-force-reload'
    }, sep, {
      label: _i18next.default.t('Go to Home'),
      role: 'homescreen',
      id: 'original-home-screen',
      click: () => {
        mainWindow?.loadURL(startUrl);
      }
    }, {
      label: _i18next.default.t('Go Back'),
      role: 'back',
      id: 'original-back',
      accelerator: getAcceleratorForPlatform('left'),
      enabled: false,
      click: () => {
        mainWindow?.webContents.goBack();
      }
    }, {
      label: _i18next.default.t('Go Forward'),
      role: 'forward',
      id: 'original-forward',
      accelerator: getAcceleratorForPlatform('right'),
      enabled: false,
      click: () => {
        mainWindow?.webContents.goForward();
      }
    }]
  }, {
    label: _i18next.default.t('Window'),
    id: 'original-window',
    submenu: [{
      label: _i18next.default.t('Minimize'),
      role: 'minimize',
      id: 'original-minimize'
    }, ...(isMac ? [sep, {
      label: _i18next.default.t('Bring All to Front'),
      role: 'front',
      id: 'original-front'
    }, sep, {
      label: _i18next.default.t('Window'),
      role: 'window',
      id: 'original-window'
    }] : [{
      label: _i18next.default.t('Close'),
      role: 'close',
      id: 'original-close'
    }])]
  }, {
    label: _i18next.default.t('Help'),
    role: 'help',
    id: 'original-help',
    afterPlugins: true,
    submenu: [{
      label: _i18next.default.t('Documentation'),
      id: 'original-documentation',
      url: 'https://headlamp.dev/docs/latest/'
    }, {
      label: _i18next.default.t('Open an Issue'),
      id: 'original-open-issue',
      url: 'https://github.com/headlamp-k8s/headlamp/issues'
    }, {
      label: _i18next.default.t('About'),
      id: 'original-about',
      url: 'https://github.com/headlamp-k8s/headlamp'
    }]
  }];
  return appMenu;
}
let loadFullMenu = false;
let currentMenu = [];
function setMenu(appWindow, newAppMenu = []) {
  let appMenu = newAppMenu;
  if (appMenu?.length === 0) {
    appMenu = getDefaultAppMenu();
  }
  let menu;
  try {
    const menuTemplate = menusToTemplate(appWindow, appMenu) || [];
    menu = _electron.Menu.buildFromTemplate(menuTemplate);
  } catch (e) {
    console.error(`Failed to build menus from template ${appMenu}:`, e);
    return;
  }
  currentMenu = appMenu;
  _electron.Menu.setApplicationMenu(menu);
}
function updateMenuLabels(menus) {
  let menusToProcess = getDefaultAppMenu();
  const defaultMenusObj = {};

  // Add all default menus in top levels and in submenus to an object:
  // id -> menu.
  while (menusToProcess.length > 0) {
    const menu = menusToProcess.shift();
    // Do not process menus that have no ids, otherwise we cannot be
    // sure which one is which.
    if (!menu.id) {
      continue;
    }
    defaultMenusObj[menu.id] = menu;
    if (menu.submenu) {
      menusToProcess = [...menusToProcess, ...menu.submenu];
    }
  }

  // Add all current menus in top levels and in submenus to a list.
  menusToProcess = [...menus];
  const menusList = [];
  while (menusToProcess.length > 0) {
    const menu = menusToProcess.shift();
    menusList.push(menu);
    if (menu.submenu) {
      menusToProcess = [...menusToProcess, ...menu.submenu];
    }
  }

  // Replace all labels with default labels if the default and current
  // menu ids are the same.
  menusList.forEach(menu => {
    if (!!menu.label && defaultMenusObj[menu.id]) {
      menu.label = defaultMenusObj[menu.id].label;
    }
  });
}
function menusToTemplate(mainWindow, menusFromPlugins) {
  const menusToDisplay = [];
  menusFromPlugins.forEach(appMenu => {
    const {
      url,
      afterPlugins = false,
      ...otherProps
    } = appMenu;
    const menu = otherProps;
    if (!loadFullMenu && !!afterPlugins) {
      return;
    }
    if (!!url) {
      menu.click = async () => {
        // Open external links in the external browser.
        if (!!mainWindow && !url.startsWith('http')) {
          mainWindow.webContents.loadURL(url);
        } else {
          await _electron.shell.openExternal(url);
        }
      };
    }

    // If the menu has a submenu, then recursively convert it.
    if (Array.isArray(otherProps.submenu)) {
      menu.submenu = menusToTemplate(mainWindow, otherProps.submenu);
    }
    menusToDisplay.push(menu);
  });
  return menusToDisplay;
}
async function getRunningHeadlampPIDs() {
  const processes = await (0, _findProcess.default)('name', 'headlamp-server.*');
  if (processes.length === 0) {
    return null;
  }
  return processes.map(pInfo => pInfo.pid);
}
function killProcess(pid) {
  if (process.platform === 'win32') {
    // Otherwise on Windows the process will stick around.
    (0, _child_process.execSync)('taskkill /pid ' + pid + ' /T /F');
  } else {
    process.kill(pid, 'SIGHUP');
  }
}
function startElecron() {
  console.info('App starting...');
  let appVersion;
  if (isDev && process.env.HEADLAMP_APP_VERSION) {
    appVersion = process.env.HEADLAMP_APP_VERSION;
    console.debug(`Overridding app version to ${appVersion}`);
  } else {
    appVersion = _electron.app.getVersion();
  }
  console.log('Check for updates: ', shouldCheckForUpdates);
  async function createWindow() {
    // WSL has a problem with full size window placement, so make it smaller.
    const withMargin = isWSL();
    const {
      width,
      height
    } = (0, _windowSize.default)(_electron.screen.getPrimaryDisplay().workAreaSize, withMargin);
    mainWindow = new _electron.BrowserWindow({
      width,
      height,
      webPreferences: {
        nodeIntegration: false,
        contextIsolation: true,
        preload: `${__dirname}/preload.js`
      }
    });
    setMenu(mainWindow, currentMenu);
    mainWindow.webContents.setWindowOpenHandler(({
      url
    }) => {
      // allow all urls starting with app startUrl to open in electron
      if (url.startsWith(startUrl)) {
        return {
          action: 'allow'
        };
      }
      // otherwise open url in a browser and prevent default
      _electron.shell.openExternal(url);
      return {
        action: 'deny'
      };
    });
    mainWindow.webContents.on('did-start-navigation', () => {
      const navigateMenu = _electron.Menu.getApplicationMenu()?.getMenuItemById('original-navigate')?.submenu;
      const goBackMenu = navigateMenu?.getMenuItemById('original-back');
      if (!!goBackMenu) {
        goBackMenu.enabled = mainWindow?.webContents.canGoBack() || false;
      }
      const goForwardMenu = navigateMenu?.getMenuItemById('original-forward');
      if (!!goForwardMenu) {
        goForwardMenu.enabled = mainWindow?.webContents.canGoForward() || false;
      }
    });
    mainWindow.webContents.on('dom-ready', () => {
      const defaultMenu = getDefaultAppMenu();
      const currentMenu = JSON.parse(JSON.stringify(defaultMenu));
      mainWindow?.webContents.send('currentMenu', currentMenu);
    });
    mainWindow.on('closed', () => {
      mainWindow = null;
    });

    // Force Single Instance Application
    const gotTheLock = _electron.app.requestSingleInstanceLock();
    if (gotTheLock) {
      _electron.app.on('second-instance', () => {
        // Someone tried to run a second instance, we should focus our window.
        if (mainWindow) {
          if (mainWindow.isMinimized()) mainWindow.restore();
          mainWindow.focus();
        }
      });
    } else {
      _electron.app.quit();
      return;
    }

    /*
    if a library is trying to open a url other than app url in electron take it
    to the default browser
    */
    mainWindow.webContents.on('will-navigate', (event, url) => {
      if (url.startsWith(startUrl)) {
        return;
      }
      event.preventDefault();
      _electron.shell.openExternal(url);
    });
    _electron.app.on('open-url', (event, url) => {
      mainWindow?.focus();
      let urlObj;
      try {
        urlObj = new URL(url);
      } catch (e) {
        _electron.dialog.showErrorBox(_i18next.default.t('Invalid URL'), _i18next.default.t('Application opened with an invalid URL: {{ url }}', {
          url
        }));
        return;
      }
      const urlParam = urlObj.hostname;
      let baseUrl = startUrl;
      // this check helps us to avoid adding multiple / to the startUrl when appending the incoming url to it
      if (baseUrl.endsWith('/')) {
        baseUrl = baseUrl.slice(0, startUrl.length - 1);
      }
      // load the index.html from build and route to the hostname received in the protocol handler url
      mainWindow?.loadURL(baseUrl + '#' + urlParam + urlObj.search);
    });
    _i18next.default.on('languageChanged', () => {
      updateMenuLabels(currentMenu);
      setMenu(mainWindow, currentMenu);
    });
    _electron.ipcMain.on('appConfig', () => {
      mainWindow?.webContents.send('appConfig', {
        checkForUpdates: shouldCheckForUpdates,
        appVersion
      });
    });
    _electron.ipcMain.on('pluginsLoaded', () => {
      loadFullMenu = true;
      console.info('Plugins are loaded. Loading full menu.');
      setMenu(mainWindow, currentMenu);
      if (pathInfoDebug && mainWindow) {
        _electron.dialog.showMessageBoxSync(mainWindow, {
          type: 'info',
          title: 'Path debug info',
          message: JSON.stringify(pathInfo)
        });
      }
    });
    _electron.ipcMain.on('setMenu', (event, menus) => {
      if (!mainWindow) {
        return;
      }

      // We don't even process this call if we're running in headless mode.
      if (isHeadlessMode) {
        console.log('Ignoring menu change from plugins because of headless mode.');
        return;
      }

      // Ignore the menu change if we received null.
      if (!menus) {
        console.log('Ignoring menu change from plugins because null was sent.');
        return;
      }

      // We update the menu labels here in case the language changed between the time
      // the original menu was sent to the renderer and the time it was received here.
      updateMenuLabels(menus);
      setMenu(mainWindow, menus);
    });
    _electron.ipcMain.on('locale', (event, newLocale) => {
      if (!!newLocale && _i18next.default.language !== newLocale) {
        _i18next.default.changeLanguage(newLocale);
      }
    });
    _electron.ipcMain.on('request-backend-token', () => {
      mainWindow?.webContents.send('backend-token', backendToken);
    });

    /**
     * Data sent from the renderer process when a 'run-command' event is emitted.
     */

    /**
     * Handles 'run-command' events from the renderer process.
     *
     * Spawns the requested command and sends 'command-stdout',
     * 'command-stderr', and 'command-exit' events back to the renderer
     * process with the command's output and exit code.
     *
     * @param event - The event object.
     * @param eventData - The data sent from the renderer process.
     */
    function handleRunCommand(event, eventData) {
      return; // Disable this until we figure out a better way to do this

      // Only allow "minikube", and "az" commands
      const validCommands = ['minikube', 'az'];
      if (!validCommands.includes(eventData.command)) {
        console.error(`Invalid command: ${eventData.command}, only valid commands are: ${JSON.stringify(validCommands)}`);
        return;
      }
      const child = (0, _child_process.spawn)(eventData.command, eventData.args, {
        ...eventData.options,
        shell: false
      });
      child.stdout.on('data', data => {
        event.sender.send('command-stdout', eventData.id, data.toString());
      });
      child.stderr.on('data', data => {
        event.sender.send('command-stderr', eventData.id, data.toString());
      });
      child.on('exit', code => {
        event.sender.send('command-exit', eventData.id, code);
      });
    }
    _electron.ipcMain.on('run-command', handleRunCommand);
    new PluginManagerEventListeners().setupEventHandlers();
    if (!useExternalServer) {
      const runningHeadlamp = await getRunningHeadlampPIDs();
      let shouldWaitForKill = true;
      if (!!runningHeadlamp) {
        const resp = _electron.dialog.showMessageBoxSync(mainWindow, {
          // Avoiding mentioning Headlamp here because it may run under a different name depending on branding (plugins).
          title: _i18next.default.t('Another process is running'),
          message: _i18next.default.t('Looks like another process is already running. Continue by terminating that process automatically, or quit?'),
          type: 'question',
          buttons: [_i18next.default.t('Continue'), _i18next.default.t('Quit')]
        });
        if (resp === 0) {
          runningHeadlamp.forEach(pid => {
            try {
              killProcess(pid);
            } catch (e) {
              console.log(`Failed to quit headlamp-servere:`, e.message);
              shouldWaitForKill = false;
            }
          });
        } else {
          mainWindow.close();
          return;
        }
      }

      // If we reach here, then we attempted to kill headlamp-server. Let's make sure it's killed
      // before starting our own, or else we may end up in a race condition (failing to start the
      // new one before the existing one is fully killed).
      if (!!runningHeadlamp && shouldWaitForKill) {
        let stillRunning = true;
        let timeWaited = 0;
        const maxWaitTime = 3000; // ms
        // @todo: Use an iterative back-off strategy for the wait (so we can start by waiting for shorter times).
        for (let tries = 1; timeWaited < maxWaitTime && stillRunning; tries++) {
          console.debug(`Checking if Headlamp is still running after we asked it to be killed; ${tries} ${timeWaited}/${maxWaitTime}ms wait.`);

          // Wait (10 * powers of 2) ms with a max of 250 ms
          const waitTime = Math.min(10 * tries ** 2, 250); // ms
          await new Promise(f => setTimeout(f, waitTime));
          timeWaited += waitTime;
          stillRunning = !!(await getRunningHeadlampPIDs());
          console.debug(stillRunning ? 'Still running...' : 'No longer running!');
        }
      }

      // If we couldn't kill the process, warn the user and quit.
      const processes = await getRunningHeadlampPIDs();
      if (!!processes) {
        _electron.dialog.showMessageBoxSync({
          type: 'warning',
          title: _i18next.default.t('Failed to quit the other running process'),
          message: _i18next.default.t(`Could not quit the other running process, PIDs: {{ process_list }}. Please stop that process and relaunch the app.`, {
            process_list: processes
          })
        });
        mainWindow.close();
        return;
      }
      serverProcess = await startServer();
      attachServerEventHandlers(serverProcess);
    }

    // Finally load the frontend
    mainWindow.loadURL(startUrl);
  }
  if (disableGPU) {
    console.info('Disabling GPU hardware acceleration. Reason: related flag is set.');
  } else if (disableGPU === undefined && process.platform === 'linux' && ['arm', 'arm64'].includes(process.arch)) {
    console.info('Disabling GPU hardware acceleration. Reason: known graphical issues in Linux on ARM (use --disable-gpu=false to force it if needed).');
    disableGPU = true;
  }
  if (disableGPU) {
    _electron.app.disableHardwareAcceleration();
  }
  _electron.app.on('ready', createWindow);
  _electron.app.on('activate', function () {
    if (mainWindow === null) {
      createWindow();
    }
  });
  _electron.app.once('window-all-closed', _electron.app.quit);
  _electron.app.once('before-quit', () => {
    _i18next.default.off('languageChanged');
    if (mainWindow) {
      mainWindow.removeAllListeners('close');
    }
  });
}
_electron.app.on('quit', quitServerProcess);

/**
 * add some error handlers to the serverProcess.
 * @param  {ChildProcess} serverProcess to attach the error handlers to.
 */
function attachServerEventHandlers(serverProcess) {
  serverProcess.on('error', err => {
    console.error(`server process failed to start: ${err}`);
  });
  serverProcess.stdout.on('data', data => {
    console.info(`server process stdout: ${data}`);
  });
  serverProcess.stderr.on('data', data => {
    const sterrMessage = `server process stderr: ${data}`;
    if (data && data.indexOf && data.indexOf('Requesting') !== -1) {
      // The server prints out urls it's getting, which aren't errors.
      console.info(sterrMessage);
    } else {
      console.error(sterrMessage);
    }
  });
  serverProcess.on('close', (code, signal) => {
    const closeMessage = `server process process exited with code:${code} signal:${signal}`;
    if (!intentionalQuit) {
      // @todo: message mainWindow, or loadURL to an error url?
      console.error(closeMessage);
    } else {
      console.info(closeMessage);
    }
    serverProcessQuit = true;
  });
}
if (isHeadlessMode) {
  serverProcess = startServer(['-html-static-dir', _path.default.join(process.resourcesPath, './frontend')]);
  attachServerEventHandlers(serverProcess);
  _electron.shell.openExternal(`http://localhost:${defaultPort}`);
} else {
  startElecron();
}