/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSInputStream;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsClientConfigKeys;
import org.apache.hadoop.hdfs.protocol.DatanodeInfoWithStorage;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.util.Time;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class TestDFSInputStreamBlockLocations {
    private static final int BLOCK_SIZE = 0x100000;
    private static final String[] RACKS = new String[]{"/d1/r1", "/d1/r1", "/d1/r2", "/d1/r2", "/d1/r2", "/d2/r3", "/d2/r3"};
    private static final int NUM_DATA_NODES = RACKS.length;
    private static final short REPLICATION_FACTOR = 4;
    private final int staleInterval = 8000;
    private final int numOfBlocks = 24;
    private final int fileLength = 0x1800000;
    private final int dfsClientPrefetchSize = 0xC00000;
    private final long dfsInputLocationsTimeout = 3600000L;
    private HdfsConfiguration conf;
    private MiniDFSCluster dfsCluster;
    private DFSClient dfsClient;
    private DistributedFileSystem fs;
    private Path filePath;
    private boolean enableBlkExpiration;

    @Parameterized.Parameters(name="{index}: CacheExpirationConfig(Enable {0})")
    public static Collection<Object[]> getTestParameters() {
        return Arrays.asList({Boolean.TRUE}, {Boolean.FALSE});
    }

    public TestDFSInputStreamBlockLocations(Boolean enableExpiration) {
        this.enableBlkExpiration = enableExpiration;
    }

    @Before
    public void setup() throws IOException {
        this.conf = new HdfsConfiguration();
        this.conf.setBoolean("dfs.namenode.avoid.read.stale.datanode", true);
        this.conf.setLong("dfs.namenode.stale.datanode.interval", 8000L);
        this.conf.setInt("dfs.namenode.heartbeat.recheck-interval", 4000);
        this.conf.setBoolean(HdfsClientConfigKeys.Read.ShortCircuit.KEY, false);
        this.conf.setInt("dfs.replication", 4);
        this.conf.setLong("dfs.blocksize", 0x100000L);
        this.conf.setLong("dfs.client.read.prefetch.size", 0xC00000L);
        if (this.enableBlkExpiration) {
            this.conf.setLong("dfs.client.refresh.read-block-locations.ms", 3600000L);
        }
        this.dfsCluster = new MiniDFSCluster.Builder((Configuration)this.conf).numDataNodes(NUM_DATA_NODES).racks(RACKS).build();
        this.dfsCluster.waitActive();
        Assert.assertEquals((long)NUM_DATA_NODES, (long)this.dfsCluster.getDataNodes().size());
        InetSocketAddress addr = new InetSocketAddress("localhost", this.dfsCluster.getNameNodePort());
        this.dfsClient = new DFSClient(addr, (Configuration)this.conf);
        this.fs = this.dfsCluster.getFileSystem();
    }

    @After
    public void teardown() throws IOException {
        if (this.dfsClient != null) {
            this.dfsClient.close();
            this.dfsClient = null;
        }
        if (this.fs != null) {
            this.fs.deleteOnExit(this.filePath);
            this.fs.close();
            this.fs = null;
        }
        if (this.dfsCluster != null) {
            this.dfsCluster.shutdown();
            this.dfsCluster = null;
        }
    }

    @Test
    public void testRefreshBlockLocations() throws IOException {
        String fileName = "/test_cache_locations";
        this.filePath = this.createFile("/test_cache_locations");
        try (DFSInputStream fin = this.dfsClient.open("/test_cache_locations");){
            LocatedBlocks existing = fin.locatedBlocks;
            long lastRefreshedAt = fin.getLastRefreshedBlocksAtForTesting();
            Assert.assertFalse((String)"should not have attempted refresh", (boolean)fin.refreshBlockLocations(null));
            Assert.assertEquals((String)"should not have updated lastRefreshedAt", (long)lastRefreshedAt, (long)fin.getLastRefreshedBlocksAtForTesting());
            Assert.assertSame((String)"should not have modified locatedBlocks", (Object)existing, (Object)fin.locatedBlocks);
            fin.addToLocalDeadNodes(this.dfsClient.datanodeReport(HdfsConstants.DatanodeReportType.LIVE)[0]);
            Assert.assertTrue((String)"should have attempted refresh", (boolean)fin.refreshBlockLocations(null));
            this.verifyChanged(fin, existing, lastRefreshedAt);
            lastRefreshedAt = fin.getLastRefreshedBlocksAtForTesting();
            existing = fin.locatedBlocks;
            HashMap<String, InetSocketAddress> mockAddressCache = new HashMap<String, InetSocketAddress>();
            InetSocketAddress unresolved = InetSocketAddress.createUnresolved("www.google.com", 80);
            for (DataNode dataNode : this.dfsCluster.getDataNodes()) {
                mockAddressCache.put(dataNode.getDatanodeUuid(), unresolved);
            }
            Assert.assertTrue((String)"should have attempted refresh", (boolean)fin.refreshBlockLocations(mockAddressCache));
            this.verifyChanged(fin, existing, lastRefreshedAt);
        }
    }

    private void verifyChanged(DFSInputStream fin, LocatedBlocks existing, long lastRefreshedAt) {
        Assert.assertTrue((String)"lastRefreshedAt should have incremented", (fin.getLastRefreshedBlocksAtForTesting() > lastRefreshedAt ? 1 : 0) != 0);
        Assert.assertNotSame((String)"located blocks should have changed", (Object)existing, (Object)fin.locatedBlocks);
        Assert.assertTrue((String)"deadNodes should be empty", (boolean)fin.getLocalDeadNodes().isEmpty());
    }

    @Test
    public void testDeferredRegistrationStatefulRead() throws IOException {
        this.testWithRegistrationMethod(DFSInputStream::read);
    }

    @Test
    public void testDeferredRegistrationPositionalRead() throws IOException {
        this.testWithRegistrationMethod(fin -> fin.readFully(0L, new byte[1]));
    }

    @Test
    public void testDeferredRegistrationGetAllBlocks() throws IOException {
        this.testWithRegistrationMethod(DFSInputStream::getAllBlocks);
    }

    @Test
    public void testClearIgnoreListChooseDataNode() throws IOException {
        String fileName = "/test_cache_locations";
        this.filePath = this.createFile("/test_cache_locations");
        try (DFSInputStream fin = this.dfsClient.open("/test_cache_locations");){
            LocatedBlocks existing = fin.locatedBlocks;
            LocatedBlock block = existing.getLastLocatedBlock();
            ArrayList<DatanodeInfoWithStorage> ignoreList = new ArrayList<DatanodeInfoWithStorage>(Arrays.asList(block.getLocations()));
            Assert.assertNotNull((Object)fin.chooseDataNode(block, ignoreList, true));
            Assert.assertEquals((long)0L, (long)ignoreList.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void testWithRegistrationMethod(ThrowingConsumer registrationMethod) throws IOException {
        String fileName = "/test_cache_locations";
        this.filePath = this.createFile("/test_cache_locations");
        DFSInputStream fin = null;
        try {
            fin = this.dfsClient.open("/test_cache_locations");
            Assert.assertFalse((String)"should not be tracking input stream on open", (boolean)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin));
            registrationMethod.accept(fin);
            Assert.assertFalse((String)"should not be tracking input stream after first read", (boolean)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin));
            fin.setLastRefreshedBlocksAtForTesting(Time.monotonicNow() - 3600001L);
            registrationMethod.accept(fin);
            Assert.assertEquals((String)"SHOULD be tracking input stream on read after interval, only if enabled", (Object)this.enableBlkExpiration, (Object)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin));
        }
        finally {
            if (fin != null) {
                fin.close();
                Assert.assertFalse((boolean)this.dfsClient.getLocatedBlockRefresher().isInputStreamTracked(fin));
            }
            this.fs.delete(this.filePath, true);
        }
    }

    private Path createFile(String fileName) throws IOException {
        Path path = new Path(fileName);
        try (FSDataOutputStream fout = this.fs.create(path, (short)4);){
            fout.write(new byte[0x1800000]);
        }
        return path;
    }

    @FunctionalInterface
    static interface ThrowingConsumer {
        public void accept(DFSInputStream var1) throws IOException;
    }
}

